// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/FinalState.hh"

namespace Rivet {


  /// @brief EEC for a wide range of energies
  class PLUTO_1981_I156315 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(PLUTO_1981_I156315);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projection
      declare(FinalState(), "FS");

      // Book histograms
      size_t ih = 1;
      for (double eVal : {7.7, 9.4, 12., 13., 17., 22., 27.6}) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en],  1, 1, ih);
        book(_c[en], "TMP/weightSum_"+en);
        ++ih;
      }
      book(_h["-"],  1, 1, ih);
      book(_c["-"], "TMP/weightSum");
      book(_h["lo"], 4, 1, 1);
      book(_h["hi"], 5, 1, 1);
      if (_sqs.empty() && inRange(sqrtS(), 30.0, 31.6)) _sqs = "-";
      raiseBeamErrorIf(_sqs.empty());

      if (_sqs == "7700"s || _sqs == "9400"s) _aeec = "lo";
      else if (_sqs == "27600"s || _sqs == "-"s) _aeec = "hi";

    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const FinalState& fs = apply<FinalState>(event, "FS");
      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (fs.particles().size() < 2) {
        MSG_DEBUG("Failed leptonic event cut");
        vetoEvent;
      }
      MSG_DEBUG("Passed leptonic event cut");
      _c[_sqs]->fill();

      const double Evis = sqr(sum(fs.particles(), Kin::E, 0.0));
      // (A)EEC
      // Need iterators since second loop starts at current outer loop iterator, i.e. no "foreach" here!
      for (Particles::const_iterator p_i = fs.particles().begin(); p_i != fs.particles().end(); ++p_i) {
        for (Particles::const_iterator p_j = p_i; p_j != fs.particles().end(); ++p_j) {
          const Vector3 mom3_i = p_i->mom().p3();
          const Vector3 mom3_j = p_j->mom().p3();
          const double energy_i = p_i->mom().E();
          const double energy_j = p_j->mom().E();
          const double thetaij = mom3_i.unit().angle(mom3_j.unit())/M_PI*180.;
          double eec = (energy_i*energy_j) / Evis;
          if (p_i != p_j) eec *= 2.;
          _h[_sqs]->fill(thetaij,  eec);
          if (!_aeec.empty()) {
            if (thetaij < 90.) {
              _h[_aeec]->fill(thetaij, -eec);
            }
            else {
              _h[_aeec]->fill(180.-thetaij, eec);
            }
          }
        }
      }

    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (const auto& item : _c) {
        const double w = item.second->val();
        if (isZero(w)) continue;
        scale(_h[item.first], 360.0/M_PI/w);
      }
      // If both energies have been run, then the merging behaviour is not well defined
      const double losum = _c["7700"s]->sumW() + _c["9400"s]->sumW();
      if (!isZero(losum))  scale(_h["lo"], 360.0/M_PI/losum);
      const double hisum = _c["27600"s]->sumW() + _c["-"s]->sumW();
      if (!isZero(hisum))  scale(_h["hi"], 360.0/M_PI/hisum);
    }

    /// @}

    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    map<string,CounterPtr> _c;
    string _sqs = "", _aeec = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(PLUTO_1981_I156315);

}
