//! Utility module to create trampolines in/out WebAssembly module.

mod func;
mod global;
mod memory;
mod table;

pub use self::func::*;
pub use self::global::*;
pub(crate) use memory::MemoryCreatorProxy;

use self::memory::create_memory;
use self::table::create_table;
use crate::prelude::*;
use crate::runtime::vm::{
    Imports, ModuleRuntimeInfo, OnDemandInstanceAllocator, SharedMemory, VMFunctionImport,
};
use crate::store::{AllocateInstanceKind, InstanceId, StoreOpaque};
use crate::{MemoryType, TableType};
use alloc::sync::Arc;
use wasmtime_environ::{MemoryIndex, Module, TableIndex, VMSharedTypeIndex};

fn create_handle(
    module: Module,
    store: &mut StoreOpaque,
    func_imports: &[VMFunctionImport],
    one_signature: Option<VMSharedTypeIndex>,
) -> Result<InstanceId> {
    let mut imports = Imports::default();
    imports.functions = func_imports;

    unsafe {
        let allocator =
            OnDemandInstanceAllocator::new(store.engine().config().mem_creator.clone(), 0, false);
        let module = Arc::new(module);
        store.allocate_instance(
            AllocateInstanceKind::Dummy {
                allocator: &allocator,
            },
            &ModuleRuntimeInfo::bare_maybe_imported_func(module, one_signature),
            imports,
        )
    }
}

pub fn generate_memory_export(
    store: &mut StoreOpaque,
    m: &MemoryType,
    preallocation: Option<&SharedMemory>,
) -> Result<crate::runtime::vm::ExportMemory> {
    let instance = create_memory(store, m, preallocation)?;
    Ok(store
        .instance_mut(instance)
        .get_exported_memory(MemoryIndex::from_u32(0)))
}

pub fn generate_table_export(
    store: &mut StoreOpaque,
    t: &TableType,
) -> Result<crate::runtime::vm::ExportTable> {
    let instance = create_table(store, t)?;
    Ok(store
        .instance_mut(instance)
        .get_exported_table(TableIndex::from_u32(0)))
}
