/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.spatial.ingest;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.geo.GeometryParserFormat;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.geometry.Geometry;
import org.elasticsearch.geometry.utils.Geohash;
import org.elasticsearch.h3.H3;
import org.elasticsearch.ingest.AbstractProcessor;
import org.elasticsearch.ingest.ConfigurationUtils;
import org.elasticsearch.ingest.IngestDocument;
import org.elasticsearch.ingest.Processor;
import org.elasticsearch.search.aggregations.bucket.geogrid.GeoTileUtils;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentFactory;
import org.elasticsearch.xcontent.XContentType;
import org.elasticsearch.xpack.spatial.common.H3CartesianUtil;

public final class GeoGridProcessor
extends AbstractProcessor {
    public static final String TYPE = "geo_grid";
    private final FieldConfig config;
    private final boolean ignoreMissing;
    private final GeometryParserFormat targetFormat;
    private final TileFieldType tileFieldType;

    GeoGridProcessor(String tag, String description, FieldConfig config, boolean ignoreMissing, GeometryParserFormat targetFormat, TileFieldType tileFieldType) {
        super(tag, description);
        this.config = config;
        this.ignoreMissing = ignoreMissing;
        this.targetFormat = targetFormat;
        this.tileFieldType = tileFieldType;
    }

    public IngestDocument execute(IngestDocument ingestDocument) {
        Object obj = ingestDocument.getFieldValue(this.config.field, Object.class, this.ignoreMissing);
        if (obj == null && this.ignoreMissing) {
            return ingestDocument;
        }
        if (obj == null) {
            throw new IllegalArgumentException("field [" + this.config.field + "] is null, cannot process it.");
        }
        if (!(obj instanceof String)) {
            throw new IllegalArgumentException("field [" + this.config.field + "] must be a String tile address");
        }
        try {
            int precision;
            List<String> nonChildren;
            List<String> children;
            String parent;
            TileHandler handler = switch (this.tileFieldType) {
                default -> throw new IncompatibleClassChangeError();
                case TileFieldType.GEOTILE -> new GeotileHandler(obj.toString());
                case TileFieldType.GEOHASH -> new GeohashHandler(obj.toString());
                case TileFieldType.GEOHEX -> new GeohexHandler(obj.toString());
            };
            Geometry geometry = handler.makeGeometry();
            XContentBuilder newValueBuilder = XContentFactory.jsonBuilder().startObject().field("val");
            this.targetFormat.toXContent(geometry, newValueBuilder, ToXContent.EMPTY_PARAMS);
            newValueBuilder.endObject();
            Map newObj = (Map)XContentHelper.convertToMap((BytesReference)BytesReference.bytes((XContentBuilder)newValueBuilder), (boolean)true, (XContentType)XContentType.JSON).v2();
            ingestDocument.setFieldValue(this.config.targetField, newObj.get("val"));
            if (this.config.parentField != null && this.config.parentField.length() > 0 && (parent = handler.getParent()) != null && parent.length() > 0) {
                ingestDocument.setFieldValue(this.config.parentField, (Object)parent);
            }
            if (this.config.childrenField != null && this.config.childrenField.length() > 0 && (children = handler.makeChildren()).size() > 0) {
                ingestDocument.setFieldValue(this.config.childrenField, children);
            }
            if (this.config.nonChildrenField != null && this.config.nonChildrenField.length() > 0 && (nonChildren = handler.makeNonChildren()).size() > 0) {
                ingestDocument.setFieldValue(this.config.nonChildrenField, nonChildren);
            }
            if (this.config.precisionField != null && this.config.precisionField.length() > 0 && (precision = handler.getPrecision()) >= 0) {
                ingestDocument.setFieldValue(this.config.precisionField, (Object)precision);
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("invalid tile definition", e);
        }
        return ingestDocument;
    }

    public String getType() {
        return TYPE;
    }

    String field(String key) {
        return this.config.field(key);
    }

    GeometryParserFormat targetFormat() {
        return this.targetFormat;
    }

    TileFieldType tileType() {
        return this.tileFieldType;
    }

    public static class FieldConfig {
        private final String field;
        private final String targetField;
        private final String parentField;
        private final String childrenField;
        private final String nonChildrenField;
        private final String precisionField;

        FieldConfig(String field, String targetField, String parentField, String childrenField, String nonChildrenField, String precisionField) {
            this.field = field;
            this.targetField = targetField;
            this.parentField = parentField;
            this.childrenField = childrenField;
            this.nonChildrenField = nonChildrenField;
            this.precisionField = precisionField;
        }

        public String field(String key) {
            return switch (key) {
                case "field" -> this.field;
                case "target_field" -> this.targetField;
                case "parent_field" -> this.parentField;
                case "children_field" -> this.childrenField;
                case "non_children_field" -> this.nonChildrenField;
                case "precision_field" -> this.precisionField;
                default -> throw new IllegalArgumentException("Invalid geo_grid processor field type [" + key + "]");
            };
        }
    }

    static enum TileFieldType {
        GEOHASH,
        GEOTILE,
        GEOHEX;


        public static TileFieldType parse(String value, String tag) {
            EnumSet<TileFieldType> validValues = EnumSet.allOf(TileFieldType.class);
            try {
                return TileFieldType.valueOf(value.toUpperCase(Locale.ROOT));
            }
            catch (IllegalArgumentException e) {
                throw ConfigurationUtils.newConfigurationException((String)GeoGridProcessor.TYPE, (String)tag, (String)"tile_type", (String)("illegal value [" + value + "], valid values are " + Arrays.toString(validValues.toArray())));
            }
        }
    }

    static class GeotileHandler
    extends TileHandler {
        private final String address;
        private final int zoom;
        private final int x;
        private final int y;

        GeotileHandler(String address) {
            this.address = address;
            int[] hashAsInts = GeoTileUtils.parseHash((String)address);
            this.zoom = hashAsInts[0];
            this.x = hashAsInts[1];
            this.y = hashAsInts[2];
        }

        @Override
        Geometry makeGeometry() {
            return GeoTileUtils.toBoundingBox((String)this.address);
        }

        @Override
        List<String> makeChildren() {
            int z = this.zoom + 1;
            if (z > 29) {
                return Collections.emptyList();
            }
            int xo = this.x * 2;
            int yo = this.y * 2;
            return Arrays.asList(GeotileHandler.hash(z, xo, yo), GeotileHandler.hash(z, xo + 1, yo), GeotileHandler.hash(z, xo, yo + 1), GeotileHandler.hash(z, xo + 1, yo + 1));
        }

        private static String hash(int zoom, int x, int y) {
            return zoom + "/" + x + "/" + y;
        }

        @Override
        String getParent() {
            return this.zoom == 0 ? "" : GeotileHandler.hash(this.zoom - 1, this.x / 2, this.y / 2);
        }

        @Override
        int getPrecision() {
            return this.zoom;
        }
    }

    static class GeohashHandler
    extends TileHandler {
        private final String hash;

        GeohashHandler(String hash) {
            this.hash = hash;
        }

        @Override
        Geometry makeGeometry() {
            return Geohash.toBoundingBox((String)this.hash);
        }

        @Override
        String getParent() {
            return this.hash.length() == 0 ? "" : this.hash.substring(0, this.hash.length() - 1);
        }

        @Override
        List<String> makeChildren() {
            return Arrays.asList(Geohash.getSubGeohashes((String)this.hash));
        }

        @Override
        int getPrecision() {
            return this.hash.length();
        }
    }

    static class GeohexHandler
    extends TileHandler {
        private final long h3;

        GeohexHandler(String spec) {
            this.h3 = H3.stringToH3((String)spec);
        }

        @Override
        Geometry makeGeometry() {
            return GeohexHandler.makePolygonFromH3(this.h3);
        }

        @Override
        String getParent() {
            int res = H3.getResolution((long)this.h3);
            if (res == 0) {
                return "";
            }
            long parent = H3.h3ToParent((long)this.h3);
            return H3.h3ToString((long)parent);
        }

        @Override
        List<String> makeChildren() {
            return GeohexHandler.asStringList(H3.h3ToChildren((long)this.h3));
        }

        @Override
        List<String> makeNonChildren() {
            return GeohexHandler.asStringList(H3.h3ToNoChildrenIntersecting((long)this.h3));
        }

        @Override
        int getPrecision() {
            return H3.getResolution((long)this.h3);
        }

        private static Geometry makePolygonFromH3(long h3) {
            return H3CartesianUtil.getNormalizeGeometry(h3);
        }

        private static List<String> asStringList(long[] cells) {
            ArrayList<String> addresses = new ArrayList<String>(cells.length);
            for (long cell : cells) {
                addresses.add(H3.h3ToString((long)cell));
            }
            return addresses;
        }
    }

    static abstract class TileHandler {
        TileHandler() {
        }

        abstract Geometry makeGeometry();

        abstract List<String> makeChildren();

        List<String> makeNonChildren() {
            return Collections.emptyList();
        }

        abstract int getPrecision();

        abstract String getParent();
    }

    public static final class Factory
    implements Processor.Factory {
        public GeoGridProcessor create(Map<String, Processor.Factory> registry, String processorTag, String description, Map<String, Object> config) {
            String field = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"field");
            String targetField = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"target_field", (String)field);
            String parentField = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"parent_field", (String)"");
            String childrenField = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"children_field", (String)"");
            String nonChildrenField = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"non_children_field", (String)"");
            String precisionField = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"precision_field", (String)"");
            boolean ignoreMissing = ConfigurationUtils.readBooleanProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"ignore_missing", (boolean)false);
            TileFieldType tileFieldType = TileFieldType.parse(ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)"tile_type"), processorTag);
            GeometryParserFormat targetFormat = this.getTargetFormat(processorTag, config);
            FieldConfig fields = new FieldConfig(field, targetField, parentField, childrenField, nonChildrenField, precisionField);
            return new GeoGridProcessor(processorTag, description, fields, ignoreMissing, targetFormat, tileFieldType);
        }

        private GeometryParserFormat getTargetFormat(String processorTag, Map<String, Object> config) {
            String propertyName = "target_format";
            String targetFormat = ConfigurationUtils.readStringProperty((String)GeoGridProcessor.TYPE, (String)processorTag, config, (String)propertyName, (String)GeometryParserFormat.GEOJSON.name());
            return switch (targetFormat.toLowerCase(Locale.ROOT).trim()) {
                case "geojson" -> GeometryParserFormat.GEOJSON;
                case "wkt" -> GeometryParserFormat.WKT;
                default -> throw ConfigurationUtils.newConfigurationException((String)GeoGridProcessor.TYPE, (String)processorTag, (String)propertyName, (String)("illegal value [" + targetFormat + "], valid values are [WKT, GEOJSON]"));
            };
        }
    }
}

