/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.plugins;

import java.io.IOException;
import java.net.URL;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Build;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.jdk.JarHell;
import org.elasticsearch.plugins.PluginBundle;
import org.elasticsearch.plugins.PluginDescriptor;

public class PluginsUtils {
    private static final Logger logger = LogManager.getLogger(PluginsUtils.class);

    private PluginsUtils() {
        throw new AssertionError((Object)"This utility class should never be instantiated");
    }

    public static List<Path> findPluginDirs(Path rootPath) throws IOException {
        ArrayList<Path> plugins = new ArrayList<Path>();
        HashSet<String> seen = new HashSet<String>();
        if (Files.exists(rootPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(rootPath);){
                for (Path plugin : stream) {
                    String filename = plugin.getFileName().toString();
                    if (FileSystemUtils.isDesktopServicesStore(plugin) || filename.startsWith(".removing-") || filename.equals(".elasticsearch-plugins.yml.cache")) continue;
                    if (!seen.add(filename)) {
                        throw new IllegalStateException("duplicate plugin: " + String.valueOf(plugin));
                    }
                    plugins.add(plugin);
                }
            }
        }
        return plugins;
    }

    public static void verifyCompatibility(PluginDescriptor info) {
        String currentVersion = Build.current().version();
        Matcher buildVersionMatcher = SemanticVersion.semanticPattern.matcher(currentVersion);
        if (buildVersionMatcher.matches()) {
            SemanticVersion currentElasticsearchSemanticVersion;
            try {
                currentElasticsearchSemanticVersion = new SemanticVersion(Integer.parseInt(buildVersionMatcher.group(1)), Integer.parseInt(buildVersionMatcher.group(2)), Integer.parseInt(buildVersionMatcher.group(3)));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Couldn't parse integers from build version [" + currentVersion + "]", e);
            }
            if (info.isStable()) {
                SemanticVersion pluginElasticsearchSemanticVersion;
                Matcher pluginEsVersionMatcher = SemanticVersion.semanticPattern.matcher(info.getElasticsearchVersion());
                if (!pluginEsVersionMatcher.matches()) {
                    throw new IllegalArgumentException("Expected semantic version for plugin [" + info.getName() + "] but was [" + info.getElasticsearchVersion() + "]");
                }
                try {
                    pluginElasticsearchSemanticVersion = new SemanticVersion(Integer.parseInt(pluginEsVersionMatcher.group(1)), Integer.parseInt(pluginEsVersionMatcher.group(2)), Integer.parseInt(pluginEsVersionMatcher.group(3)));
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Expected integer version for plugin [" + info.getName() + "] but found [" + info.getElasticsearchVersion() + "]", e);
                }
                if (pluginElasticsearchSemanticVersion.major != currentElasticsearchSemanticVersion.major) {
                    throw new IllegalArgumentException("Stable Plugin [" + info.getName() + "] was built for Elasticsearch major version " + pluginElasticsearchSemanticVersion.major + " but version " + currentVersion + " is running");
                }
                if (pluginElasticsearchSemanticVersion.after(currentElasticsearchSemanticVersion)) {
                    throw new IllegalArgumentException("Stable Plugin [" + info.getName() + "] was built for Elasticsearch version " + info.getElasticsearchVersion() + " but earlier version " + currentVersion + " is running");
                }
            } else if (!info.getElasticsearchVersion().equals(currentVersion)) {
                throw new IllegalArgumentException("Plugin [" + info.getName() + "] was built for Elasticsearch version " + info.getElasticsearchVersion() + " but version " + currentVersion + " is running");
            }
        }
        JarHell.checkJavaVersion((String)info.getName(), (String)info.getJavaVersion());
    }

    public static void checkForFailedPluginRemovals(Path pluginsDirectory) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(pluginsDirectory, ".removing-*");){
            Iterator<Path> iterator = stream.iterator();
            if (iterator.hasNext()) {
                Path removing = iterator.next();
                String fileName = removing.getFileName().toString();
                String name = fileName.substring(1 + fileName.indexOf(45));
                String message = String.format(Locale.ROOT, "found file [%s] from a failed attempt to remove the plugin [%s]; execute [elasticsearch-plugin remove %2$s]", removing, name);
                throw new IllegalStateException(message);
            }
        }
    }

    static Set<PluginBundle> getModuleBundles(Path modulesDirectory) throws IOException {
        return PluginsUtils.findBundles(modulesDirectory, "module");
    }

    static Set<PluginBundle> getPluginBundles(Path pluginsDirectory) throws IOException {
        return PluginsUtils.findBundles(pluginsDirectory, "plugin");
    }

    public static Map<String, List<String>> getDependencyMapView(Path pluginsDirectory) throws IOException {
        return PluginsUtils.getPluginBundles(pluginsDirectory).stream().collect(Collectors.toMap(b -> b.plugin.getName(), b -> b.plugin.getExtendedPlugins()));
    }

    private static Set<PluginBundle> findBundles(Path directory, String type) throws IOException {
        HashSet<PluginBundle> bundles = new HashSet<PluginBundle>();
        for (Path plugin : PluginsUtils.findPluginDirs(directory)) {
            PluginBundle bundle = PluginsUtils.readPluginBundle(plugin, type);
            if (!bundles.add(bundle)) {
                throw new IllegalStateException("duplicate " + type + ": " + String.valueOf(bundle.plugin));
            }
            if (!type.equals("module") || !bundle.plugin.getName().startsWith("test-") || Build.current().isSnapshot()) continue;
            throw new IllegalStateException("external test module [" + String.valueOf(plugin.getFileName()) + "] found in non-snapshot build");
        }
        logger.trace(() -> "findBundles(" + type + ") returning: " + String.valueOf(bundles.stream().map(b -> b.plugin.getName()).sorted().toList()));
        return bundles;
    }

    private static PluginBundle readPluginBundle(Path plugin, String type) throws IOException {
        PluginDescriptor info;
        try {
            info = PluginDescriptor.readFromProperties(plugin);
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not load plugin descriptor for " + type + " directory [" + String.valueOf(plugin.getFileName()) + "]", e);
        }
        return new PluginBundle(info, plugin);
    }

    public static void preInstallJarHellCheck(PluginDescriptor candidateInfo, Path candidateDir, Path pluginsDir, Path modulesDir, Set<URL> classpath) throws IOException {
        HashSet<PluginBundle> bundles = new HashSet<PluginBundle>(PluginsUtils.getPluginBundles(pluginsDir));
        bundles.addAll(PluginsUtils.getModuleBundles(modulesDir));
        bundles.add(new PluginBundle(candidateInfo, candidateDir));
        List<PluginBundle> sortedBundles = PluginsUtils.sortBundles(bundles);
        HashMap<String, Set<URL>> transitiveUrls = new HashMap<String, Set<URL>>();
        for (PluginBundle bundle : sortedBundles) {
            PluginsUtils.checkBundleJarHell(classpath, bundle, transitiveUrls);
        }
    }

    static void checkBundleJarHell(Set<URL> systemLoaderURLs, PluginBundle bundle, Map<String, Set<URL>> transitiveUrls) {
        List<String> exts = bundle.plugin.getExtendedPlugins();
        try {
            Logger logger = LogManager.getLogger(JarHell.class);
            HashSet<URL> extendedPluginUrls = new HashSet<URL>();
            for (String extendedPlugin : exts) {
                Set<URL> pluginUrls = transitiveUrls.get(extendedPlugin);
                assert (pluginUrls != null) : "transitive urls should have already been set for " + extendedPlugin;
                extendedPluginUrls.addAll(pluginUrls);
                JarHell.checkJarHell(extendedPluginUrls, arg_0 -> ((Logger)logger).debug(arg_0));
                HashSet<URL> intersection = new HashSet<URL>(bundle.allUrls);
                intersection.retainAll(pluginUrls);
                if (!intersection.isEmpty()) {
                    throw new IllegalStateException("jar hell! duplicate codebases with extended plugin [" + extendedPlugin + "]: " + String.valueOf(intersection));
                }
                HashSet<URL> implementation = new HashSet<URL>(bundle.allUrls);
                implementation.addAll(extendedPluginUrls);
                JarHell.checkJarHell(implementation, arg_0 -> ((Logger)logger).debug(arg_0));
            }
            extendedPluginUrls.addAll(bundle.getExtensionUrls());
            transitiveUrls.put(bundle.plugin.getName(), extendedPluginUrls);
            HashSet<URL> intersection = new HashSet<URL>(systemLoaderURLs);
            intersection.retainAll(bundle.allUrls);
            if (!intersection.isEmpty()) {
                throw new IllegalStateException("jar hell! duplicate codebases between plugin and core: " + String.valueOf(intersection));
            }
            HashSet<URL> union = new HashSet<URL>(systemLoaderURLs);
            union.addAll(bundle.allUrls);
            JarHell.checkJarHell(union, arg_0 -> ((Logger)logger).debug(arg_0));
        }
        catch (IllegalStateException ise) {
            throw new IllegalStateException("failed to load plugin " + bundle.plugin.getName() + " due to jar hell", ise);
        }
        catch (Exception e) {
            throw new IllegalStateException("failed to load plugin " + bundle.plugin.getName() + " while checking for jar hell", e);
        }
    }

    static List<PluginBundle> sortBundles(Set<PluginBundle> bundles) {
        Map<String, PluginBundle> namedBundles = bundles.stream().collect(Collectors.toMap(b -> b.plugin.getName(), Function.identity()));
        LinkedHashSet<PluginBundle> sortedBundles = new LinkedHashSet<PluginBundle>();
        LinkedHashSet<String> dependencyStack = new LinkedHashSet<String>();
        for (PluginBundle bundle : bundles) {
            PluginsUtils.addSortedBundle(bundle, namedBundles, sortedBundles, dependencyStack);
        }
        return new ArrayList<PluginBundle>(sortedBundles);
    }

    private static void addSortedBundle(PluginBundle bundle, Map<String, PluginBundle> bundles, LinkedHashSet<PluginBundle> sortedBundles, LinkedHashSet<String> dependencyStack) {
        String name = bundle.plugin.getName();
        if (dependencyStack.contains(name)) {
            StringBuilder msg = new StringBuilder("Cycle found in plugin dependencies: ");
            dependencyStack.forEach(s -> {
                msg.append((String)s);
                msg.append(" -> ");
            });
            msg.append(name);
            throw new IllegalStateException(msg.toString());
        }
        if (sortedBundles.contains(bundle)) {
            return;
        }
        dependencyStack.add(name);
        for (String dependency : bundle.plugin.getExtendedPlugins()) {
            PluginBundle depBundle = bundles.get(dependency);
            if (depBundle == null) {
                throw new IllegalArgumentException("Missing plugin [" + dependency + "], dependency of [" + name + "]");
            }
            PluginsUtils.addSortedBundle(depBundle, bundles, sortedBundles, dependencyStack);
            assert (sortedBundles.contains(depBundle));
        }
        dependencyStack.remove(name);
        sortedBundles.add(bundle);
    }

    private record SemanticVersion(int major, int minor, int bugfix) {
        static final Pattern semanticPattern = Pattern.compile("^(\\d+)\\.(\\d+)\\.(\\d+)$");

        boolean after(SemanticVersion other) {
            if (this.major < other.major) {
                return false;
            }
            if (this.major > other.major) {
                return true;
            }
            if (this.minor < other.minor) {
                return false;
            }
            if (this.minor > other.minor) {
                return true;
            }
            return this.bugfix > other.bugfix;
        }

        @Override
        public String toString() {
            return Strings.format("%d.%d.%d", this.major, this.minor, this.bugfix);
        }
    }
}

