package io.oversec.one.crypto.encoding;

import io.oversec.one.common.CoreContract;
import io.oversec.one.crypto.CoreContractTestBase;
import io.oversec.one.crypto.Utils;
import io.oversec.one.crypto.proto.Outer;
import io.oversec.one.crypto.symsimple.SimpleSymmetricCryptoHandler;

import org.junit.Before;
import org.junit.Test;
public class ZeroWidthXCoderTest extends EncodingTestBase {

    @Before
    public void setUp() throws Exception {
        CoreContract.Companion.init(new CoreContractTestBase() {
            @Override
            public boolean isDbSpreadInvisibleEncoding(String packagename) {
                return true;
            }
        });
        super.setUp();
    }

    @Override
    AbstractXCoder createCoder() {
        return new ZeroWidthXCoder(mContext);
    }

    @Test
    public void testEncodeDecodeGpg() throws Exception {
        Outer.Msg msgIn = createTestOuterMsgPgp();
        String encoded = mCoder.encodeInternal(msgIn,null,"foo.bar");
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgIn, msgOut);
    }

    @Test
    public void testEncodeDecodeGpgPreKotlinization() throws Exception {
        String encoded = new String(Utils.hexToBytes("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"),"utf-8");
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgOut.getMsgTextGpgV0().getCiphertext().toStringUtf8(), CONTENT);
    }

    @Test
    public void testEncodeDecodeGpgWithDecoyText() throws Exception {
        Outer.Msg msgIn = createTestOuterMsgPgp();
        String encoded = mCoder.encodeInternal(msgIn,null,"foo.bar") + "DecoyDecoyDecoy";
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgIn, msgOut);
    }

    @Test
    public void testEncodeDecodeGpgWithDecoyTextPreKotlinization() throws Exception {
        String encoded = new String(Utils.hexToBytes("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"),"utf-8");
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgOut.getMsgTextGpgV0().getCiphertext().toStringUtf8(), CONTENT);
    }


    @Test
    public void testEncodeDecodeSym() throws Exception {
        Outer.Msg msgIn = createTestOuterMsgSym();
        String encoded = mCoder.encodeInternal(msgIn,null,"foo.bar");
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgIn, msgOut);
    }

    @Test
    public void testEncodeDecodeSymPreKotlinization() throws Exception {
        String encoded = new String(Utils.hexToBytes("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"),"utf-8");
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgOut.getMsgTextSymSimpleV0().getMsgTextChaChaV0().getCostKeyhash(), SimpleSymmetricCryptoHandler.KEY_ID_COST);
        assertEquals(1,msgOut.getMsgTextSymSimpleV0().getMsgTextChaChaV0().getPerKeyCiphertextCount());
        Outer.MsgTextChaChaV0_KeyAndSaltAndCiphertext c1 = msgOut.getMsgTextSymSimpleV0().getMsgTextChaChaV0().getPerKeyCiphertext(0);
        assertEquals(c1.getSalt().toStringUtf8(),SALT);
        assertEquals(c1.getIv().toStringUtf8(),IV);
        assertEquals(c1.getCiphertext().toStringUtf8(),CONTENT);
    }

    @Test
    public void testEncodeDecodeSymWithDecoyText() throws Exception {
        Outer.Msg msgIn = createTestOuterMsgSym();
        String encoded = mCoder.encodeInternal(msgIn,null,"foo.bar") + "DecoyDecoyDecoy";
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgIn, msgOut);
    }

    @Test
    public void testEncodeDecodeSymWithDecoyTextPreKotlinization() throws Exception {
        String encoded = new String(Utils.hexToBytes("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"),"utf-8");
        Outer.Msg msgOut = mCoder.decode(encoded);
        assertEquals(msgOut.getMsgTextSymSimpleV0().getMsgTextChaChaV0().getCostKeyhash(), SimpleSymmetricCryptoHandler.KEY_ID_COST);
        assertEquals(1,msgOut.getMsgTextSymSimpleV0().getMsgTextChaChaV0().getPerKeyCiphertextCount());
        Outer.MsgTextChaChaV0_KeyAndSaltAndCiphertext c1 = msgOut.getMsgTextSymSimpleV0().getMsgTextChaChaV0().getPerKeyCiphertext(0);
        assertEquals(c1.getSalt().toStringUtf8(),SALT);
        assertEquals(c1.getIv().toStringUtf8(),IV);
        assertEquals(c1.getCiphertext().toStringUtf8(),CONTENT);
    }
}