/*
 * campfire, formerly known as bandcampDirect
 * Copyright (C) 2020 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.activity;

import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.os.Bundle;
import android.os.Handler;
import android.view.View;
import android.widget.ImageView;
import android.widget.TextView;
import androidx.annotation.Nullable;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import com.android.volley.Response;
import com.android.volley.toolbox.Volley;
import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.SharedPreferences;
import godau.fynn.bandcampdirect.ZoomImage;
import godau.fynn.bandcampdirect.adapter.ArtistAdapter;
import godau.fynn.bandcampdirect.bandcamp.Bandcamp;
import godau.fynn.bandcampdirect.model.Artist;

public class ArtistActivity extends Activity {

    public static final String EXTRA_ARTIST = "artist";
    public static final String ARTIST_ID = "id";

    private final ZoomImage zoomImage = new ZoomImage();
    private Bandcamp bandcamp;


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_artist);

        SharedPreferences sharedPreferences = new SharedPreferences(ArtistActivity.this);
        bandcamp = new Bandcamp(sharedPreferences.createUser());


        if (getIntent().hasExtra(EXTRA_ARTIST)) {

            // View artist

            Artist artist = (Artist) getIntent().getSerializableExtra(EXTRA_ARTIST);

            displayArtist(artist);

            displayCoverArt(artist.getImage());
        } else {

            // Download artist from ID
            bandcamp.postArtistDetails(getIntent().getLongExtra(ARTIST_ID, -1), new Handler(msg -> {

                Artist artist = (Artist) msg.getData().getSerializable(Bandcamp.ARTIST);

                displayArtist(artist);

                displayCoverArt(artist.getImage());

                return false;
            }), new Handler(msg -> {
                showInfo("A network error occurred");
                return false;
            }));

        }
    }

    private void displayArtist(Artist artist) {

        displayMeta(artist.getName(), artist.getLocation());

        RecyclerView recyclerView = findViewById(R.id.recyclerView);
        recyclerView.setLayoutManager(new LinearLayoutManager(ArtistActivity.this));

        recyclerView.setAdapter(new ArtistAdapter(artist));
    }

    private void displayMeta(String title, String artist) {
        // Title text view
        TextView titleView = findViewById(R.id.title);
        titleView.setText(title);

        // Artist text view
        if (artist != null) {
            ((TextView) findViewById(R.id.artist)).setText(artist);
        }

        // Show divider
        findViewById(R.id.divider).setVisibility(View.VISIBLE);
    }

    private void showInfo(@Nullable String info) {
        if (info != null) {
            findViewById(R.id.infoLayout).setVisibility(View.VISIBLE);
            ((TextView) findViewById(R.id.info)).setText(info);
        } else {
            findViewById(R.id.infoLayout).setVisibility(View.GONE);
        }
    }

    private void displayCoverArt(String cover) {
        // Cover art

        ImageView coverView = findViewById(R.id.cover);

        Volley.newRequestQueue(ArtistActivity.this).add(Bandcamp.makeCoverArtRequest(
                cover, new Response.Listener<Bitmap>() {
                    @Override
                    public void onResponse(Bitmap response) {
                        BitmapDrawable bitmapDrawable = new BitmapDrawable(response);
                        coverView.setImageDrawable(bitmapDrawable);

                        coverView.setOnClickListener(new View.OnClickListener() {
                            @Override
                            public void onClick(View view) {
                                zoomImage.zoomImage(coverView, bitmapDrawable,
                                        findViewById(R.id.expandedCover), findViewById(R.id.root));
                            }
                        });
                    }
                }, null));
    }
}
