//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.network;

import fil.libre.repwifiapp.Commons;
import fil.libre.repwifiapp.helpers.Logger;
import fil.libre.repwifiapp.helpers.RootCommand;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

public abstract class Engine implements IEngine {
    
    public static final String PIDFILE_DHCPCD = "/data/misc/dhcp/dhcpcd-wlan0.pid";
        
    
    @Override
    public AccessPointInfo[] getAvailableNetworks() {

        Logger.logDebug("getAvailableNetworks():");

        if (!WpaSupplicant.start()) {
            Logger.logError("Failed starting wpa_supplicant");
            return null;
        }

        if (!WpaCli.scanNetworks()) {
            Logger.logError("failed scanning networks");
            return null;
        }

        String scanRes = WpaCli.getScanResults();

        if (scanRes == null) {
            Logger.logError("failed getting scan results");
            return null;
        }

        AccessPointInfo[] a = AccessPointInfo.parseScanResult(scanRes);
        if (a == null) {
            Logger.logError("Unable to parse scan file into AccessPointInfo array");
            return a;
        }

        Logger.logDebug("# of APs found: " + a.length);

        return a;

    }
    
    public ConnectionStatus getConnectionStatus() {
        
        ConnectionStatus s = WpaCli.getConnectionStatus();
        if (s == null){
            return null;
        }
        
        String ifcfg = getIfconfigString();
        s.parseIfconfigOutput(ifcfg);
        return s;        
    }

    private String getIfconfigString(){
        
        // needs root for accessing encapsulation and hardware address (tested).
        RootCommand cmd = new RootCommand("ifconfig " + WpaSupplicant.INTERFACE_NAME);
        
        try {
            if (cmd.execute() != 0){
                Logger.logError("FAILED to run ifconfig to obtain interface info.");
                return null;
            }
        } catch (Exception e) {
            Logger.logError("Exception while running ifconfig.", e);
            return null;
        }
        
        return cmd.getOutput();        
        
    }

    @Override
    public abstract int connect(AccessPointInfo info);

    @Override
    public boolean disconnect() {

        return WpaCli.disconnect();

    }

    public static boolean isInterfaceAvailable(String ifaceName) throws SocketException {

        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            NetworkInterface nif = interfaces.nextElement();
            String nm = nif.getName();
            if (nm.equals(ifaceName)) {
                return true;
            }
        }
        return false;

    }

    public int runDhcpcd() throws Exception{
        // needs root
        // option -w avoids dhcpcd forking to background,
        // in order to keep control over its exit code, and be able to wait for it.
        // option -A avoids ARP IP checking, we use it to save some seconds in the connection process.
        // option -t <timeout> sets time out for obtaining a lease.
        String cmdText = "dhcpcd -w -A -t " + Commons.WAIT_FOR_DHCPCD + " " + WpaSupplicant.INTERFACE_NAME;
        RootCommand su = new RootCommand(cmdText);
        return su.execute();
    }
    
    public int runDhcpcd(DhcpSettings dhcpConfig) throws Exception{
        
        if (dhcpConfig == null || dhcpConfig.useDhcp){
            Logger.logDebug("running dhchpc without dhcp settings, reverting to dhcp");
            return runDhcpcd();
        }
        
        Logger.logDebug("Running dhcpcd with custom ip settings");
        String cmdMask = "dhcpcd -w -A -S ip_address=%s -S routers=%s -t %d %s";
        String cmd = String.format(cmdMask,
                                    dhcpConfig.getStaticIPwithMask(),
                                    dhcpConfig.getDefaultGateway(),
                                    Commons.WAIT_FOR_DHCPCD,
                                    WpaSupplicant.INTERFACE_NAME);
        
        RootCommand su = new RootCommand(cmd);
        return su.execute();
        
    }

    public static boolean killDhcpcd(){
        if (! RootCommand.executeRootCmd("killall -SIGINT dhcpcd")){
            return false;
        }
        return RootCommand.executeRootCmd("rm " + PIDFILE_DHCPCD);
    }
    
    public boolean interfaceUp() {
        // needs root (tested)
        return RootCommand.executeRootCmd("ifconfig " + WpaSupplicant.INTERFACE_NAME + " up");
    }

   
}
