//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.network;

import android.nfc.FormatException;
import android.os.Parcel;
import android.os.Parcelable;
import fil.libre.repwifiapp.Utils;
import fil.libre.repwifiapp.helpers.Logger;
import org.apache.http.conn.util.InetAddressUtils;
import org.json.JSONException;
import org.json.JSONObject;

public class DhcpSettings implements Parcelable {

    public boolean useDhcp;
    private String _staticIP;
    private int _mask;
    private String _defGw;

    private static final String JSONKEY_DHCP = "DHCP";
    private static final String JSONKEY_STATIC_IP = "StaticIP";
    private static final String JSONKEY_GW = "Gateway";

    public DhcpSettings(boolean useDhcp, String staticIP, String subnetMask, String defaultGateway)
                    throws FormatException {
        this(useDhcp, staticIP, Utils.netmaskStringToInt(subnetMask), defaultGateway);
    }

    public DhcpSettings(boolean useDhcp, String staticIP, int subnetMask, String defaultGatweay)
                    throws FormatException {

        this.useDhcp = useDhcp;

        if (!useDhcp) {

            if (!validateParams(staticIP, defaultGatweay, subnetMask)) {
                throw new FormatException("Invalid dhcp parameters!");
            }

            this._staticIP = staticIP;
            this._mask = subnetMask;
            this._defGw = defaultGatweay;

        }

    }

    private DhcpSettings() {
        // inner use
    }

    public static DhcpSettings parseSavedSettings(String staticIPwithMask, String defaultGateway) {

        try {

            String[] ipm = staticIPwithMask.split("/");
            String ip = ipm[0];
            int mask = Integer.parseInt(ipm[1]);

            return new DhcpSettings(false, ip, mask, defaultGateway);

        } catch (Exception e) {
            Logger.logError("Exception while parsing DhcpSettings for saved network. Reverting to dhcp.",
                            e);
            return null;
        }

    }

    public static DhcpSettings getDefault() {
        try {
            return new DhcpSettings(true, null, 24, null);
        } catch (FormatException e) {
            // no format exception can happen.
            return null;
        }
    }

    private boolean validateParams(String ip, String gateway, int mask) {

        if (isValidAddress(ip) && isValidAddress(gateway) && isValidMask(mask)) {
            return true;
        } else {
            return false;
        }

    }

    public static boolean isValidAddress(String ipAddress) {
        return InetAddressUtils.isIPv4Address(ipAddress);
    }

    public static boolean isValidMaks(String mask) {
        int m = Utils.netmaskStringToInt(mask);
        return isValidMask(m);
    }

    public static boolean isValidMask(int mask) {
        if (mask >= 8 && mask <= 32) {
            return true;
        } else {
            return false;
        }
    }

    public String getStaticIP() {
        if (_staticIP == null) {
            return "";
        }
        return _staticIP;
    }

    public String getStaticIPwithMask() {
        return getStaticIP() + "/" + String.valueOf(getSubnetMaskInt());
    }

    public int getSubnetMaskInt() {
        return _mask;
    }

    public String getSubnetMaskString() {
        String v = Utils.netmaskIntToString(_mask);
        if (v == null) {
            return "";
        }
        return v;
    }

    public String getDefaultGateway() {
        if (_defGw == null) {
            return "";
        }
        return _defGw;
    }

    public JSONObject toJson() {

        JSONObject j = new JSONObject();

        try {
            j.put(JSONKEY_DHCP, useDhcp);
            j.put(JSONKEY_GW, getDefaultGateway());
            j.put(JSONKEY_STATIC_IP, getStaticIPwithMask());

            return j;

        } catch (JSONException e) {
            Logger.logError("Exception while converting DhcpSettings to JSON.", e);
            return null;
        }

    }

    public static DhcpSettings fromJsonObject(JSONObject json) {

        if (json == null) {
            return null;
        }

        DhcpSettings sets = new DhcpSettings();

        try {

            sets.useDhcp = json.getBoolean(JSONKEY_DHCP);

            if (json.has(JSONKEY_GW) && !json.isNull(JSONKEY_GW)) {
                sets._defGw = json.getString(JSONKEY_GW);
            }

            if (json.has(JSONKEY_STATIC_IP) && !json.isNull(JSONKEY_STATIC_IP)) {

                String[] splt = json.getString(JSONKEY_STATIC_IP).split("/");
                sets._staticIP = splt[0];
                sets._mask = Integer.parseInt(splt[1]);

            }

            return sets;

        } catch (Exception e) {
            Logger.logError("Exception while parsing json object to DhcpSettings", e);
            return null;
        }

    }

    @Override
    public int describeContents() {
        return 0;
    }

    public DhcpSettings(Parcel in) {

        this.useDhcp = in.readInt() == 1 ? true : false;
        this._staticIP = in.readString();
        this._mask = in.readInt();
        this._defGw = in.readString();

    }

    public static final Parcelable.Creator<DhcpSettings> CREATOR = new Parcelable.Creator<DhcpSettings>() {
        public DhcpSettings createFromParcel(Parcel in) {
            return new DhcpSettings(in);
        }

        @Override
        public DhcpSettings[] newArray(int size) {
            return new DhcpSettings[size];
        }

    };

    @Override
    public void writeToParcel(Parcel dest, int flags) {

        dest.writeInt(useDhcp ? 1 : 0);
        dest.writeString(_staticIP);
        dest.writeInt(_mask);
        dest.writeString(_defGw);

    }

}
