//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.network;

import android.os.Parcel;
import android.os.Parcelable;
import fil.libre.repwifiapp.Utils;
import fil.libre.repwifiapp.helpers.Logger;
import java.net.InetAddress;
import java.net.UnknownHostException;

public class ConnectionStatus implements Parcelable {

    public static final String STATUS_CONNECTED = "COMPLETED";
    public static final String STATUS_INACTIVE = "INACTIVE";
    public static final String STATUS_DISCONNECTED = "DISCONNECTED";
    public static final String STATUS_UNDEFINED = "UNDEFINED";

    public String wpaStatus = null;
    public String SSID = null;
    public String BSSID = null;
    public String IP = null;
    public String gateway = null;
    public String subnetMask = null;
    public String hwAddress = null;
    public String broadcastAddress = null;

    private static final String F_SEP = "=";
    private static final String KeyStatus = "wpa_state";
    private static final String KeySSID = "ssid";
    private static final String KeyBSSID = "bssid";
    private static final String KeyIP = "ip_address";

    private static final String IFCONFIG_BCAST = "Bcast";
    private static final String IFCONFIG_MASK = "Mask";
    private static final String IFCONFIG_HWADDR = "HWaddr";
    private static final String IFCONFIG_KVALSEP = ":";
    private static final String IFCONFIG_FSEP = "  ";

    public ConnectionStatus() {
    }

    public static ConnectionStatus parseWpaCliOutput(String wpaCliOutput) {

        if (wpaCliOutput == null) {
            return null;
        }

        if (wpaCliOutput.trim().length() == 0) {
            return null;
        }

        String[] lines = wpaCliOutput.split("\n");

        ConnectionStatus s = new ConnectionStatus();
        for (String line : lines) {

            if (line.trim().equals("")) {
                continue;
            }

            String[] fields = line.split(F_SEP);
            if (fields.length < 2) {
                continue;
            }

            String key = fields[0];
            String val = fields[1];

            if (key.equals(KeyBSSID)) {
                s.BSSID = val;
            } else if (key.equals(KeySSID)) {
                s.SSID = val;
            } else if (key.equals(KeyStatus)) {
                s.wpaStatus = val;
            } else if (key.equals(KeyIP)) {
                s.IP = val;
            }

        }

        return s;

    }

    public static ConnectionStatus getDummyDisconnected() {
        ConnectionStatus s = new ConnectionStatus();
        s.wpaStatus = STATUS_DISCONNECTED;
        return s;
    }

    public boolean parseIfconfigOutput(String ifconfigOutput) {

        if (ifconfigOutput == null) {
            return false;
        }

        String[] lines = ifconfigOutput.split("\n");
        if (lines.length < 2) {
            return false;
        }

        String[] fields1 = lines[0].split(IFCONFIG_FSEP);
        String[] fields2 = lines[1].split(IFCONFIG_FSEP);

        for (String f : fields1) {
            // first line uses a single blank space as key-val separator
            String[] splt = f.split(" ");
            if (splt.length == 2 && splt[0].equals(IFCONFIG_HWADDR)) {
                this.hwAddress = splt[1];
            }
        }

        for (String f : fields2) {
            String[] splt = f.split(IFCONFIG_KVALSEP);
            if (splt.length == 2) {

                String key = splt[0];
                String val = splt[1];

                if (key.equals(IFCONFIG_MASK)) {

                    this.subnetMask = val.trim();

                } else if (key.equals(IFCONFIG_BCAST)) {

                    this.broadcastAddress = val;

                } else if (key.equals(IFCONFIG_HWADDR)) {

                    this.hwAddress = val;

                }

            }
        }

        return true;

    }

    public boolean isConnected() {

        if (this.wpaStatus == null) {
            return false;
        }

        if (this.wpaStatus.equals(STATUS_CONNECTED)) {
            return true;
        } else {
            return false;
        }
    }

    /***
     * @return Returns a string representation of the current IP address and
     *         subnet mask (e.g. "192.168.1.123/24"); returns null if either the
     *         address or the mask is null.
     */
    public String addressAndMaskToString() {

        if (this.IP == null || this.subnetMask == null) {
            return null;
        }

        int mask = Utils.netmaskStringToInt(this.subnetMask);
        return this.IP + "/" + mask;

    }

    public InetAddress getInetAddress() {
        if (this.IP == null) {
            return null;
        }

        try {
            return InetAddress.getByName(this.IP);
        } catch (UnknownHostException e) {
            Logger.logError("Exception while parsing InetAddress from string", e);
            return null;
        }

    }

    public int getSubnetMaskInt() {
        return Utils.netmaskStringToInt(this.subnetMask);
    }

    public InetAddress getGatewayInetAddress() {

        if (this.gateway == null) {
            return null;
        }

        try {
            return InetAddress.getByName(this.gateway);
        } catch (Exception e) {
            Logger.logError("Exception while parsing gateway's InetAddress from string.", e);
            return null;
        }

    }

    public AccessPointInfo getNetworkDetails() {
        AccessPointInfo i = new AccessPointInfo(SSID, BSSID, "", "", "");
        return NetworkManager.getSavedNetwork(i);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public ConnectionStatus(Parcel in) {

        this.SSID = in.readString();
        this.BSSID = in.readString();
        this.IP = in.readString();
        this.subnetMask = in.readString();
        this.gateway = in.readString();
        this.hwAddress = in.readString();
        this.broadcastAddress = in.readString();
        this.wpaStatus = in.readString();

    }

    public static final Parcelable.Creator<ConnectionStatus> CREATOR = new Parcelable.Creator<ConnectionStatus>() {
        public ConnectionStatus createFromParcel(Parcel in) {
            return new ConnectionStatus(in);
        }

        public ConnectionStatus[] newArray(int size) {
            return new ConnectionStatus[size];
        }
    };

    @Override
    public void writeToParcel(Parcel dest, int flags) {

        dest.writeString(this.SSID);
        dest.writeString(this.BSSID);
        dest.writeString(this.IP);
        dest.writeString(this.subnetMask);
        dest.writeString(this.gateway);
        dest.writeString(this.hwAddress);
        dest.writeString(this.broadcastAddress);
        dest.writeString(this.wpaStatus);

    }

    public boolean equals(ConnectionStatus status) {

        if (status == null) {
            return false;
        }

        if (status.isConnected() != this.isConnected()) {
            return false;
        }

        return fieldEquals(this.IP, status.IP) && fieldEquals(this.BSSID, status.BSSID)
                        && fieldEquals(this.SSID, status.SSID)
                        && fieldEquals(this.subnetMask, status.subnetMask)
                        && fieldEquals(this.gateway, status.gateway)
                        && fieldEquals(this.hwAddress, status.hwAddress);

    }

    private boolean fieldEquals(String myField, String extField) {
        return myField == null ? extField == null : myField.equals(extField);
    }

    @Override
    public String toString() {
        return String.format(
                        "WPAsts: %s; \nIP: %s; \nMask: %s; \nGway: %s; \nBcast: %s; \nHWaddr: %s ",
                        wpaStatus, IP, subnetMask, gateway, broadcastAddress, hwAddress);
    }

}
