//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.activities;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences.Editor;
import android.os.Bundle;
import android.preference.CheckBoxPreference;
import android.preference.EditTextPreference;
import android.preference.ListPreference;
import android.preference.Preference;
import android.preference.PreferenceActivity;
import android.preference.PreferenceFragment;
import android.preference.PreferenceManager;
import fil.libre.repwifiapp.Prefs;
import fil.libre.repwifiapp.R;
import fil.libre.repwifiapp.R.string;
import fil.libre.repwifiapp.Utils;
import fil.libre.repwifiapp.helpers.Logger;
import fil.libre.repwifiapp.service.ConnectionManagementService;
import org.apache.http.conn.util.InetAddressUtils;
import java.util.List;

public class SettingsActivity extends PreferenceActivity {

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        setTheme(R.style.Theme_SettingsTheme);
        super.onCreate(savedInstanceState);

    }

    @Override
    public void onBuildHeaders(List<Header> target) {
        loadHeadersFromResource(R.xml.settings_headers, target);
    }

    public static class DebugSettingFragment extends PreferenceFragment {

        @Override
        public void onCreate(Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);
            addPreferencesFromResource(R.xml.debug_settings);

            setConfirmKillBackend();
            setDumpFileClick();
            setDebugPriorityChangeListener();
        }

        private void setConfirmKillBackend() {

            Preference pref = getPreferenceScreen().findPreference("pref_kill_backend");
            pref.setOnPreferenceClickListener(new Preference.OnPreferenceClickListener() {

                @Override
                public boolean onPreferenceClick(Preference p) {
                    Utils.killBackEnd(getActivity(), false);
                    return true;
                }
            });

        }

        private void setDumpFileClick() {
            Preference pref = getPreferenceScreen().findPreference("pref_dump_log");
            pref.setOnPreferenceClickListener(new Preference.OnPreferenceClickListener() {

                @Override
                public boolean onPreferenceClick(Preference p) {
                    if (Logger.dumpLogcatToFile(getActivity())) {
                        Utils.showMessage(
                                        getString(R.string.msg_log_saved) + ": \n"
                                                        + Logger.getLogDumpFile(), getActivity());

                    } else {
                        Utils.showMessage(getString(R.string.msg_log_save_fail), getActivity());
                    }
                    return true;
                }
            });
        }

        /**
         * Monitors changes in the "log priority" setting, and reflects them
         * onto the Logger class.
         */
        private void setDebugPriorityChangeListener() {

            ListPreference logPrioPref = (ListPreference) getPreferenceScreen().findPreference(
                            "debug_priority");

            logPrioPref.setOnPreferenceChangeListener(new Preference.OnPreferenceChangeListener() {

                @Override
                public boolean onPreferenceChange(Preference preference, Object newValue) {
                    Logger.setLogPriority(Integer.parseInt((String) newValue));
                    return true;
                }
            });
        }

    }

    public static class GeneralSettingFragment extends PreferenceFragment {

        @Override
        public void onCreate(Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);
            addPreferencesFromResource(R.xml.general_settings);

            setIpFormatValidationListener(Prefs.PREF_DNS_1);
            setIpFormatValidationListener(Prefs.PREF_DNS_2);
            setConfirmRestore();
            
            formatSummaryMonitoryConnection();

        }

        private void formatSummaryMonitoryConnection() {

            CheckBoxPreference chkMonitor = (CheckBoxPreference) findPreference("monitor_connection");
            chkMonitor.setSummary(getString(string.summary_monitor_connection_state).replace(
                            ConnectionManagementService.PLACEHOLDER_CHECK_STATUS_INTERVAL,
                            String.valueOf(ConnectionManagementService.CHECK_STATUS_INTERVAL_SECS)));

        }

        private void setIpFormatValidationListener(String prefName) {
            EditTextPreference editPref = (EditTextPreference) getPreferenceScreen()
                            .findPreference(prefName);

            editPref.setOnPreferenceChangeListener(new Preference.OnPreferenceChangeListener() {

                @Override
                public boolean onPreferenceChange(Preference preference, Object newValue) {
                    
                    if (((String) newValue).isEmpty()
                                    || InetAddressUtils.isIPv4Address((String) newValue)) {
                        return true;
                    } else {
                        Utils.showMessage(getString(R.string.msg_error_ip_format), getActivity());
                        return false;
                    }
                }
            });
        }

        private void setConfirmRestore() {

            Preference pref = getPreferenceScreen().findPreference("pref_restore_default");
            pref.setOnPreferenceClickListener(new Preference.OnPreferenceClickListener() {

                @Override
                public boolean onPreferenceClick(Preference p) {
                    resetSettingsDefault(false);
                    return true;
                }
            });

        }

        public void resetSettingsDefault(boolean silent) {

            Context context = getActivity();

            if (silent) {

                Editor e = PreferenceManager.getDefaultSharedPreferences(context).edit();
                e.clear();
                e.commit();

            } else {

                String msg = context.getString(R.string.confirm_reset_settings);
                AlertDialog.Builder dlgAlert = new AlertDialog.Builder(context,
                                R.style.Theme_RepWifiDialogTheme);
                dlgAlert.setMessage(msg);
                dlgAlert.setPositiveButton(context.getString(android.R.string.ok),
                                new DialogInterface.OnClickListener() {
                                    @Override
                                    public void onClick(DialogInterface dialog, int whichButton) {
                                        resetSettingsDefault(true);
                                        return;
                                    }
                                });
                dlgAlert.setNegativeButton(context.getString(android.R.string.cancel),
                                new DialogInterface.OnClickListener() {
                                    @Override
                                    public void onClick(DialogInterface dialog, int whichButton) {
                                        return;
                                    }
                                });

                dlgAlert.setCancelable(false);
                AlertDialog diag = dlgAlert.create();

                diag.show();

                return;
            }

        }

    }

}
