//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.activities;

import android.app.AlertDialog;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.content.res.AssetManager;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.hardware.usb.UsbManager;
import android.os.Bundle;
import android.os.Parcelable;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.RelativeLayout.LayoutParams;
import android.widget.TextView;
import android.widget.Toast;
import fil.libre.repwifiapp.ActivityLauncher;
import fil.libre.repwifiapp.ActivityLauncher.RequestCode;
import fil.libre.repwifiapp.Commons;
import fil.libre.repwifiapp.Prefs;
import fil.libre.repwifiapp.R;
import fil.libre.repwifiapp.Utils;
import fil.libre.repwifiapp.helpers.Logger;
import fil.libre.repwifiapp.helpers.RootCommand;
import fil.libre.repwifiapp.network.AccessPointInfo;
import fil.libre.repwifiapp.network.ConnectionResult;
import fil.libre.repwifiapp.network.ConnectionStatus;
import fil.libre.repwifiapp.network.Engine;
import fil.libre.repwifiapp.network.NetworkManager;
import fil.libre.repwifiapp.network.WpaCli;
import fil.libre.repwifiapp.network.WpaSupplicant;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketException;

public class MainActivity extends VpnAndConnectionBoundActivity{

    private ActivityLauncher launcher = new ActivityLauncher(this);
    private BroadcastReceiver detachReceiver;
    private ConnectionStatus status = null;
    
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        Commons.init(getApplicationContext());

        if (!NetworkManager.updateStorageVersion()) {
            Logger.logError("Failed to convert storage file to new version!");
        }

        toggleStatusAppearance(false);
        setUsbDeviceMonitor();
        setVersionOnTitle();

        if (isSystemApp()) {
            setTitle(getTitle() + " (sysapp)");
        }
        
    }

    @Override
    public void onStart() {
        super.onStart();
        Logger.logDebug("Main onStart()");

        if (handleIntent()) {
            // app called for a specific intent.
            // avoid any other task.
            Log.d("RepWifi", "handleIntent returned true");
            return;
        }

        checkConditions();

        if (isServiceBound()) {
            requestStatusUpdate();
        }

        Logger.logDebug("Main onStart() returning.");
        
    }

    @Override
    public void onSaveInstanceState(Bundle savedInstanceState) {
        super.onSaveInstanceState(savedInstanceState);
    }

    private boolean handleIntent() {

        Intent i = getIntent();
        if (i != null && i.hasExtra(ActivityLauncher.EXTRA_REQCODE)) {

            switch (i.getIntExtra(ActivityLauncher.EXTRA_REQCODE, -1)) {

            case RequestCode.NONE:
                moveTaskToBack(true);

                break;

            default:
                break;
            }

            return true;

        } else {
            // no intent to handle.
            return false;
        }

    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent intent) {

        Logger.logDebug("Main onActivityResult(): ");

        if (intent == null) {
            return;
        }

        if (resultCode != RESULT_OK && requestCode != RequestCode.VPN_PERMISSION_CONN) {
            return;
        }

        AccessPointInfo i = null;
        if (intent.hasExtra(ActivityLauncher.EXTRA_APINFO)) {
            Bundle xtras = intent.getExtras();
            i = (AccessPointInfo) xtras.getParcelable(ActivityLauncher.EXTRA_APINFO);
        }

        switch (requestCode) {

        case RequestCode.PASS_INPUT:
            Logger.logDebug("ReqCode: PASS_INPUT");
            handleResultSetPass(i);
            break;

        case RequestCode.SELECT_CONN:
            Logger.logDebug("ReqCode: SELECT_CONN");
            boolean rescan = intent.getExtras().getBoolean(ActivityLauncher.EXTRA_RESCAN);
            handleResultSelect(i, rescan);
            break;

        case RequestCode.CONNECT:
            Logger.logDebug("ReqCode: CONNECT");
            ConnectionResult conres = (ConnectionResult) intent.getExtras().getParcelable(
                            ActivityLauncher.EXTRA_CONNRES);
            handleFinishedConnecting(conres, i);
            break;

        case RequestCode.NETWORKS_GET:
            Logger.logDebug("ReqCode: NETWORKS_GET");
            Parcelable[] p = intent.getExtras().getParcelableArray(
                            ActivityLauncher.EXTRA_APINFO_ARR);
            AccessPointInfo[] nets = AccessPointInfo.fromParcellableArray(p);
            launcher.launchSelectActivity(nets, true, false);
            break;

        case RequestCode.STATUS_SHOW:
            Logger.logDebug("ReqCode: STATUS_SHOW");
            // do nothing
            break;

        case RequestCode.SELECT_DETAILS:
            Logger.logDebug("ReqCode: SELECT_DETAILS");
            launcher.launchDetailsActivity(i);
            break;

        case RequestCode.DETAILS_SHOW:
            Logger.logDebug("ReqCode: DETAILS_SHOW");
            boolean del = intent.getExtras().getBoolean(ActivityLauncher.EXTRA_DELETE);
            if (del) {
                deleteNetwork(i);
            }
            break;

        case RequestCode.CONNECT_HIDDEN:
            Logger.logDebug("ReqCode: CONNECT_HIDDEN");

            if (i != null) {
                Logger.logDebug("NetworkInfo NOT null, handling result");
                handleResultSelect(i, false);
            }
            break;

        default:

            break;

        }

    }

    private void setVersionOnTitle() {

        try {

            String vers = getPackageManager().getPackageInfo(getPackageName(), 0).versionName;
            if (vers == null) {
                return;
            }

            setTitle(getTitle() + " - v." + vers);

        } catch (Exception e) {
            Logger.logError("Error while setting version on MainActivity's title.", e);
        }

    }

    private void showStatus(ConnectionStatus status) {

        Logger.logDebug("MainActivity.showStatus()");

        String msg = "";
        this.status = status;

        try {

            if (status == null) {
                msg = getString(R.string.text_status) + ": [NULL]";
                toggleStatusAppearance(false);

            } else if (this.status.isConnected()) {
                Logger.logDebug("showStatus isConnected,showing buttons");
                msg = getString(R.string.msg_connected_to) + " " + status.SSID;
                toggleStatusAppearance(true);

            } else {
                Logger.logDebug("showStatus status Else");
                msg = getString(R.string.msg_disconnected);
                toggleStatusAppearance(false);
            }

        } catch (Exception e) {
            Logger.logError("Exception on showStatus", e);
            msg = "[ERORR]";
        }

        TextView view = (TextView) findViewById(R.id.txt_status);
        view.setText(msg);

    }

    private void toggleStatusAppearance(boolean connected) {

        try {
            Button b = (Button) findViewById(R.id.btn_disconnect);
            Button i = (Button) findViewById(R.id.btn_info);
            ImageView img = (ImageView) findViewById(R.id.img_logo);

            LayoutParams lp = (LayoutParams) img.getLayoutParams();

            b.setEnabled(connected);
            i.setEnabled(connected);

            if (connected) {
                b.setVisibility(View.VISIBLE);
                i.setVisibility(View.VISIBLE);
                lp.removeRule(RelativeLayout.CENTER_HORIZONTAL);

            } else {
                b.setVisibility(View.INVISIBLE);
                i.setVisibility(View.INVISIBLE);
                lp.addRule(RelativeLayout.CENTER_HORIZONTAL);
            }

            setLogoDrawable(img, connected);
            img.setLayoutParams(lp);

        } catch (Exception e) {
            Logger.logError("Error while setting status appearance", e);

        }

    }

    BitmapDrawable logoConn;
    BitmapDrawable logoDisc;

    public void setLogoDrawable(ImageView img, boolean connected) {

        img.setImageDrawable(null);
        System.gc();

        try {

            String res = "repwifi-logo-0-small.png";
            BitmapDrawable logo = logoConn;
            if (!connected) {
                logo = logoDisc;
                res = "repwifi-logo-1-small.png";
            }

            AssetManager am = getAssets();
            InputStream s = am.open(res);

            if (logo == null) {
                logo = (BitmapDrawable) Drawable.createFromStream(s, res);
            }
            img.setImageDrawable(logo);

            s.close();
        } catch (IOException e) {
        }

    }

    public void onBtnDisconnectClick(View v) {

        disconnectVpn();
        sendCmdDisconnect();

    }

    public void onBtnInfoClick(View v) {
        launcher.launchStatusActivity(this.status);
    }

    private void setUsbDeviceMonitor() {
        detachReceiver = new BroadcastReceiver() {
            public void onReceive(Context context, Intent intent) {
                if (intent.getAction().equals(UsbManager.ACTION_USB_DEVICE_DETACHED)) {
                    handleUsbEvent(true);
                } else if (intent.getAction().equals(UsbManager.ACTION_USB_DEVICE_ATTACHED)) {
                    handleUsbEvent(false);
                }
            }
        };

        IntentFilter filter = new IntentFilter();
        filter.addAction(UsbManager.ACTION_USB_DEVICE_DETACHED);
        registerReceiver(detachReceiver, filter);

        IntentFilter filt2 = new IntentFilter();
        filt2.addAction(UsbManager.ACTION_USB_DEVICE_ATTACHED);
        registerReceiver(detachReceiver, filt2);
    }

    private boolean checkConditions(boolean serviceStarted) {
        // second chance to have the service bound:
        boolean conds = (checkRootEnabled() && checkIsSystemApp() && checkInterface(true));
        if (serviceStarted) {
            conds = conds && isServiceBound();
        }
        return conds;
    }

    private boolean checkConditions() {
        return checkConditions(false);
    }

    private boolean checkInterface(boolean alert) {

        boolean res = false;
        String msg = "";

        try {
            res = Engine.isInterfaceAvailable(WpaSupplicant.INTERFACE_NAME);
        } catch (SocketException e) {
            Logger.logError("SocketException during isInterfaceAvailable()", e);
            msg = "Error while retrieving interface list!";
            res = false;
        }

        if (res == false && alert) {
            msg = getResources().getString(R.string.msg_interface_not_found);
            Utils.showMessage(msg, this);
        }

        return res;

    }

    private boolean checkRootEnabled() {

        boolean result = false;
        String msg = "Unknown Root error";
        RootCommand su = new RootCommand(null);

        int excode = -1;

        try {
            excode = su.testRootAccess();
        } catch (Exception e) {
            Logger.logError("Error while trying to get first Super User access.", e);
            excode = -1;
            result = false;
        }

        switch (excode) {
        case 0:
            result = true;
            break;

        case Commons.EXCOD_ROOT_DENIED:
            result = false;
            msg = getResources().getString(R.string.msg_root_denied);
            break;

        case Commons.EXCOD_ROOT_DISABLED:
            result = false;
            msg = getResources().getString(R.string.msg_root_disabled);
            break;

        default:
            result = false;
            msg = "Unknown Root error.\nExit code " + excode;
            break;
        }

        if (!result) {
            Utils.showMessage(msg, this);
        }

        return result;

    }

    private void handleResultSelect(AccessPointInfo i, boolean rescan) {

        if (rescan) {

            doScan();

        } else if (i != null) {

            if (i.needsPassword()) {

                // try to fetch network's configuration from storage
                AccessPointInfo fromStorage = NetworkManager.getSavedNetwork(i);
                if (fromStorage == null) {

                    launcher.launchPasswordActivity(i);
                    return;

                } else {
                    // use fetched network
                    i = fromStorage;
                }

            }

            // disconnect any vpn before connecting to a new network
            disconnectVpn();
            launcher.launchLongTaskActivityConnect(i);
        }

    }

    private void handleResultSetPass(AccessPointInfo i) {
        launcher.launchLongTaskActivityConnect(i);
    }

    private void handleFinishedConnecting(ConnectionResult connectionResult, AccessPointInfo info) {

        Logger.logDebug("handleFinishedConnecting");

        String toastText = null;
        int res = connectionResult.getResult();
        ConnectionStatus status = connectionResult.getStatus();

        switch (res) {

        case ConnectionResult.CONN_OK:

                                Logger.logDebug("About to launch VPN on successful connection result.");
                    beginConnectVpn(status.getNetworkDetails());
                
            
            break;

        case ConnectionResult.CONN_GW_FAIL:
            Logger.logDebug("Result code CONN_GW_FAIL");
            toastText = getString(R.string.msg_gw_failed);
            break;

        case ConnectionResult.CONN_TIMEOUT:
            // probable wrong password:
            handleConnectionTimeout(info);
            break;

        default:
            Logger.logDebug("Result code: " + res);
            toastText = getString(R.string.msg_connect_fail);
            break;

        }

        if (toastText != null) {
            Toast toast = Toast.makeText(getApplicationContext(), toastText, Toast.LENGTH_LONG);
            toast.show();
        }

        showStatus(status);

    }

    private void handleConnectionTimeout(AccessPointInfo info) {
        // reset wpa_supplicant's state
        WpaCli.disconnect();
        if (!WpaCli.terminateSupplicant()) {
            WpaSupplicant.kill();
        }

        if (info.needsPassword()) {
            // prompt user for password retry:
            launcher.launchPasswordActivity(info, getString(R.string.msg_connection_timeout));
        } else {
            Utils.showMessage(getString(R.string.msg_connection_timeout_nopass), this);
        }

    }

    private void deleteNetwork(AccessPointInfo info) {

        String msg = "";
        if (NetworkManager.remove(info)) {
            msg = getString(R.string.msg_netinfo_deleted);
        } else {
            msg = getString(R.string.msg_netinfo_delete_fail);
        }

        Toast toast = Toast.makeText(this, msg, Toast.LENGTH_LONG);
        toast.show();

    }

    private void handleUsbEvent(boolean detached) {

        if (detached && !checkInterface(false)) {
            // device disconnected:
            // clear back-end state and update status.
            disconnectVpn();

        } else if (isAutoConnectEnabled()) {

            try {

                // waits for a maximum of WAIT_ON_USB_ATTACHED milliseconds
                // to let the interface being registered.
                int msWaited = 0;
                while (msWaited < Commons.WAIT_ON_USB_ATTACHED) {

                    Thread.sleep(100);
                    msWaited += 100;

                    if (checkInterface(false)) {
                        sendCmdAutoconnect();
                        return;
                    }
                }

            } catch (InterruptedException e) {
                // ignores and exits;
                return;
            }

        }

        requestStatusUpdate();

    }

    private boolean isAutoConnectEnabled() {
        return Prefs.getBoolean(getApplicationContext(), Prefs.PREF_AUTOCONNECT, false);
    }

    private void doScan() {
        if (checkConditions(true)) {
            launcher.launchLongTaskActivityScan();
        }
    }

    public void btnScanClick(View v) {
        doScan();
    }

    public void btnHiddenClick(View v) {
        if (checkConditions(true)) {
            launcher.launchInputSsidActivity();
        }
    }

    public void btnManageClick(View v) {
        launcher.launchSelectActivity(null, false, true);
    }

    private boolean checkIsSystemApp() {

        if (isSystemApp()) {
            return true;
        } else {
            promtpMakeSystemApp();
            return false;
        }
    }

    private void promtpMakeSystemApp() {

        String msg = getString(R.string.msg_make_system_app);
        AlertDialog.Builder dlgAlert = new AlertDialog.Builder(this,
                        R.style.Theme_RepWifiDialogTheme);
        dlgAlert.setMessage(msg);
        dlgAlert.setPositiveButton(getString(android.R.string.ok),
                        new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int whichButton) {
                                confirmMakeSystemApp();
                                return;
                            }
                        });
        dlgAlert.setNegativeButton(getString(android.R.string.cancel),
                        new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int whichButton) {
                                return;
                            }
                        });

        dlgAlert.setCancelable(false);
        AlertDialog diag = dlgAlert.create();

        diag.show();

    }

    private void confirmMakeSystemApp() {
        String msg = getString(R.string.msg_confirm_make_system_app);
        AlertDialog.Builder dlgAlert = new AlertDialog.Builder(this,
                        R.style.Theme_RepWifiDialogTheme);
        dlgAlert.setMessage(msg);
        dlgAlert.setPositiveButton(getString(android.R.string.yes),
                        new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int whichButton) {
                                makeSystemApp();
                                return;
                            }
                        });
        dlgAlert.setNegativeButton(getString(android.R.string.no),
                        new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int whichButton) {
                                return;
                            }
                        });

        dlgAlert.setCancelable(false);
        AlertDialog diag = dlgAlert.create();

        diag.show();
    }

    private boolean isSystemApp() {

        PackageManager pm = getPackageManager();
        boolean isSystem = false;
        try {
            ApplicationInfo ai = pm.getApplicationInfo(getPackageName(), 0);

            isSystem = ((ai.flags & ApplicationInfo.FLAG_SYSTEM) == 1);

        } catch (NameNotFoundException e) {
            // always succeeds, as we use our own package
        }
        return isSystem;

    }

    public static final String SYS_FOLDER = "/system";
    public static final String SYS_APP_FOLDER = "/system/priv-app";
    private static final String SYSAPP_SCRIPT_FNAME = "make-system-app.sh";

    private boolean makeSystemApp() {

        PackageManager pm = getPackageManager();
        String pkgName = getPackageName();
        File currentSourceDir = null;
        try {
            ApplicationInfo ai = pm.getApplicationInfo(pkgName, 0);
            currentSourceDir = new File(ai.sourceDir);

        } catch (NameNotFoundException e) {
        }// always succeeds, as we use our own package

        String parentFolder = currentSourceDir.getParentFile().getName();
        if (!parentFolder.contains(pkgName)) {
            // use the plain package name as a parent folder for the apk
            parentFolder = pkgName;
        }

        String targetDir = SYS_APP_FOLDER + "/" + parentFolder;

        // get own data directory:
        File ownDir = getFilesDir();
        File scriptFile = new File(ownDir, SYSAPP_SCRIPT_FNAME);
        String conts = Utils.rawResourceAsString(getApplicationContext(), R.raw.make_system_app);

        if (conts == null) {
            Logger.logError("Error while opening script from raw resources.");
            return false;
        }

        if (!Utils.writeFile(scriptFile.getAbsolutePath(), conts, true)) {
            Logger.logError("Failed to write script contents to file.");
            return false;
        }

        Logger.logDebug("Starting script to make myself a system app...");

        String cmd = "bash \"" + scriptFile.getAbsolutePath() + "\"" + " \"" + targetDir + "\""
                        + " \"" + currentSourceDir.getAbsolutePath() + "\"" + " \"" + SYS_FOLDER
                        + "\"";

        return RootCommand.executeRootCmd(cmd);

    }
    
        
    @Override
    protected void onMsgDisconnectReport(ConnectionStatus status) {
        String msg = "";

        if (status != null && !status.isConnected()) {
            msg = getString(R.string.msg_disconnected);
        } else {
            msg = getString(R.string.msg_disconnect_fail);
        }

        Toast.makeText(this, msg, Toast.LENGTH_LONG).show();
        showStatus(status);

    }
    
    @Override
    protected void onMsgAutoconnectReport(AccessPointInfo[] infos) {
        if (infos != null && infos.length > 0) {
            launcher.launchSelectActivity(infos, true, false);
        }
    }

    @Override
    protected void onMsgStatusChange(ConnectionStatus status) {
        Logger.logDebug("Received status update from service.");
        showStatus(status);
    }

    @Override
    protected void onManagementServiceConnected() {
        super.onManagementServiceConnected();
        requestStatusUpdate();
    }

    @Override
    protected void onStop() {
        super.onStop();
    }

    @Override
    public void onDestroy() {

        if (detachReceiver != null) {
            unregisterReceiver(detachReceiver);
        }

        super.onDestroy();
    }

}
