//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.activities;

import android.app.Activity;
import android.content.Intent;
import android.nfc.FormatException;
import android.os.Bundle;
import android.view.Menu;
import android.view.View;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.CompoundButton.OnCheckedChangeListener;
import android.widget.EditText;
import fil.libre.repwifiapp.ActivityLauncher;
import fil.libre.repwifiapp.R;
import fil.libre.repwifiapp.Utils;
import fil.libre.repwifiapp.network.AccessPointInfo;
import fil.libre.repwifiapp.network.DhcpSettings;
import fil.libre.repwifiapp.network.NetworkManager;

public class Ipv4SettingsActivity extends Activity implements OnCheckedChangeListener {

    private AccessPointInfo currentNetwork;
    private DhcpSettings currentSettings;

    private EditText txtIp;
    private EditText txtGw;
    private EditText txtMask;
    private CheckBox chkDhcp;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_ipv4_settings);

        chkDhcp = (CheckBox) findViewById(R.id.chk_use_dhcp);
        txtIp = (EditText) findViewById(R.id.txt_static_ip);
        txtMask = (EditText) findViewById(R.id.txt_netmask);
        txtGw = (EditText) findViewById(R.id.txt_gateway);

        chkDhcp.setOnCheckedChangeListener(this);

        Intent intent = getIntent();
        if (!intent.hasExtra(ActivityLauncher.EXTRA_APINFO)) {
            this.setResult(RESULT_CANCELED);
            this.finish();
            return;
        }

        this.currentNetwork = (AccessPointInfo) intent.getExtras().getParcelable(ActivityLauncher.EXTRA_APINFO);
        if (this.currentNetwork == null) {
            this.setResult(RESULT_CANCELED);
            this.finish();
            return;
        }

        this.currentNetwork = NetworkManager.getSavedNetwork(currentNetwork);
        
        loadNetwork();

    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        // getMenuInflater().inflate(R.menu.activity_manage_networks, menu);
        return true;
    }

    private void loadNetwork() {

        setTitle(getString(R.string.text_ipv4_settings) + " " + this.currentNetwork.getSsid());
        currentSettings = this.currentNetwork.getDhcpConfiguration();
        loadSettings();
        
    }

    private void loadSettings() {

        if (currentSettings.useDhcp) {

            chkDhcp.setChecked(true);

            txtIp.setText("");
            txtIp.setEnabled(false);

            txtMask.setText("");
            txtMask.setEnabled(false);

            txtGw.setText("");
            txtGw.setEnabled(false);

        } else {
            chkDhcp.setChecked(false);

            txtIp.setText(currentSettings.getStaticIP());
            txtIp.setEnabled(true);

            txtMask.setText(currentSettings.getSubnetMaskString());
            txtMask.setEnabled(true);

            txtGw.setText(currentSettings.getDefaultGateway());
            txtGw.setEnabled(true);
        }

    }

    @Override
    public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {

        if (buttonView == findViewById(R.id.chk_use_dhcp)) {
            currentSettings.useDhcp = isChecked;
            loadSettings();

        }

    }

    public void btnSaveClick(View v) throws FormatException {

        if (chkDhcp.isChecked()) {
            currentSettings.useDhcp = true;

        } else {

            String ip = txtIp.getText().toString();
            String mask = txtMask.getText().toString();
            String gw = txtGw.getText().toString();

            if (!DhcpSettings.isValidAddress(ip)) {
                Utils.showMessage(getString(R.string.msg_invalid_ip),this);
                return;
            }

            if (!DhcpSettings.isValidMaks(mask)) {
                Utils.showMessage(getString(R.string.msg_invalid_netmask),this);
                return;
            }

            if (!DhcpSettings.isValidAddress(gw)) {
                Utils.showMessage(getString(R.string.msg_invalid_gateway),this);
                return;
            }

            currentSettings = new DhcpSettings(chkDhcp.isChecked(), ip, mask, gw);

        }

        currentNetwork.setDhcpConfiguration(currentSettings);
        if (NetworkManager.save(currentNetwork)) {
            Utils.showMessage(getString(R.string.msg_network_saved),this);
        } else {
            Utils.showMessage(getString(R.string.msg_network_save_fail),this);
        }

    }

    public void btnBackClick(View v) {
        finish();
    }

}
