//
// Copyright 2017 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.helpers;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import fil.libre.repwifiapp.Commons;
import android.util.Log;

public class Utils {

    private static final long MILLIS_IN_DAY = 86400000;

    public static final String APP_NAME = "RepWifi";

    private static Exception _lastException = null;

    public static Exception getLastException() {
        return _lastException;
    }

    public static void logError(String msg, Exception e) {
        Log.e(APP_NAME, msg, e);
    }

    public static void logError(String msg) {
        Log.e(APP_NAME, msg);
    }

    public static void logDebug(String msg) {
        logDebug(msg, 0);
    }

    public static void logDebug(String msg, int level) {

        if (level < Commons.getLogPriority()) {
            return;
        }

        Log.d(APP_NAME, msg);
    }

    public static boolean writeFileLines(String filePath, String[] lines, boolean overwrite) {

        if (lines == null) {
            return false;
        }

        if (lines.length == 0) {
            return true;
        }

        StringBuilder sb = new StringBuilder();

        for (String l : lines) {

            if (l == null) {
                return false;
            }

            sb.append(l + "\n");
        }

        return writeFile(filePath, sb.toString(), overwrite);

    }

    public static boolean writeFile(String filePath, String content, boolean overwrite) {

        if (content == null) {
            return false;
        }

        FileWriter writer = null;

        try {

            writer = new FileWriter(filePath, (!overwrite));
            writer.write(content);

            return true;

        } catch (Exception e) {
            _lastException = e;
            return false;

        } finally {

            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    logError("error while closing filewriter", e);
                }
            }

        }

    }

    public static String[] readFileLines(String filePath) {

        if (filePath == null) {
            return null;
        }

        File f = new File(filePath);
        if (!f.exists()) {
            logError("File doesn't exist: " + filePath);
            return null;
        }

        FileReader fr = null;
        BufferedReader bufr = null;

        List<String> lines = new ArrayList<String>();

        try {

            fr = new FileReader(filePath);
            bufr = new BufferedReader(fr);
            String line = "";

            while ((line = bufr.readLine()) != null) {
                lines.add(line);
            }

            String[] ar = new String[lines.size()];

            return lines.toArray(ar);

        } catch (Exception e) {
            logError("Error while reading file " + filePath, e);
            return null;

        } finally {
            try {
                if (bufr != null) {
                    bufr.close();
                }
            } catch (IOException ex) {
                logError("error while closing filereader", ex);
            }
            try {
                if (fr != null) {
                    fr.close();
                }
            } catch (IOException exc) {
                logError("error while closing filereader", exc);
            }
        }

    }

    public static String readFile(String filePath){
        
        if (filePath == null) {
            return null;
        }

        File f = new File(filePath);
        if (!f.exists()) {
            logError("File doesn't exist: " + filePath);
            return null;
        }

        FileReader fr = null;
        BufferedReader bufr = null;

        StringBuilder sb = new StringBuilder();

        try {

            fr = new FileReader(filePath);
            bufr = new BufferedReader(fr);
            String line = "";

            while ((line = bufr.readLine()) != null) {
                sb.append(line);
                sb.append("\n");
            }

            return sb.toString();

        } catch (Exception e) {
            logError("Error while reading file " + filePath, e);
            return null;

        } finally {
            try {
                if (bufr != null) {
                    bufr.close();
                }
            } catch (IOException ex) {
                logError("error while closing filereader", ex);
            }
            try {
                if (fr != null) {
                    fr.close();
                }
            } catch (IOException exc) {
                logError("error while closing filereader", exc);
            }
        }
        
    }
    
    public static long daysToMilliseconds(int days) {
        return (days * MILLIS_IN_DAY);
    }

    public static long millisecondsToDays(long milliseconds) {
        return (milliseconds / MILLIS_IN_DAY);
    }

    public static boolean dumpLogcatToFile(String filePath) {

        if (filePath == null) {
            return false;
        }

        try {

            String cmd1 = "logcat -d | grep " + APP_NAME + ">" + filePath;
            String cmd2 = "logcat -d | grep " + Commons.getContext().getPackageName() + ">>"
                            + filePath;
            String SEP_LOG = "\n\n---------- [REPWIFI_LOG_SEPARATOR] ----------\n\n";

            RootCommand c1 = new RootCommand(cmd1);
            RootCommand c2 = new RootCommand(cmd2);

            if (c1.execute() != 0) {
                return false;
            }

            if (!writeFile(filePath, SEP_LOG, false)) {
                return false;
            }

            if (c2.execute() != 0) {
                return false;
            }

            return true;

        } catch (Exception e) {
            logError("Exception during log dump.", e);
            return false;
        }

    }

    public static String netmaskIntToString(int mask) {
       
        if (mask < 8 || mask > 32) {
            return null;
        }

        StringBuilder sb = new StringBuilder(32);
        StringBuilder sb2 = new StringBuilder(15);

        for (int i = 0; i < mask; i++) {
            sb.append("1");
        }

        for (int i = 0; i < 32 - mask; i++) {
            sb.append("0");
        }

        for (int i = 0; i < 3; i++) {
            String bitString = sb.substring((i * 8), (i * 8) + 8);
            int ibyte = Integer.parseInt(bitString, 2);
            sb2.append(ibyte);
            sb2.append(".");
        }
        String bitString = sb.substring(24, 32);
        int ibyte = Integer.parseInt(bitString, 2);
        sb2.append(ibyte);

        return sb2.toString();

    }

    public static int netmaskStringToInt(String mask) {

        if (mask == null) {
            return -1;
        }

        String[] octs = mask.split("\\.");
        if (octs.length != 4) {
            return -1;
        }

        int intmask = 0;
        boolean prevIsZero = false;
        for (String o : octs) {

            int intval = 0;
            
            try {
                intval = Integer.parseInt(o, 10);
            } catch (NumberFormatException e) {
                return -1;
            }            

            String b = Integer.toBinaryString(intval);
            if (b.length() != 8 && b.contains("1")) {
                //invalid mask! has ones after a zero
                return -1;
            }
            for (int i = 0; i < b.length(); i++) {
                if (b.charAt(i) == '0') {
                    prevIsZero = true;

                } else if (prevIsZero) {
                    // invalid mask
                    return -1;
                    
                } else {
                    intmask += 1;
                }
                
            }
                       
        }
        
        return intmask;

    }

}
