//
// Copyright 2017 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.helpers;

import org.apache.http.conn.util.InetAddressUtils;
import org.json.JSONException;
import org.json.JSONObject;
import java.io.Serializable;
import android.nfc.FormatException;

public class DhcpSettings implements Serializable {

    private static final long serialVersionUID = 1L;
    public boolean useDhcp;
    private String _staticIP;
    private int _mask;
    private String _defGw;

    private static final String JSONKEY_DHCP = "DHCP";
    private static final String JSONKEY_STATIC_IP = "StaticIP";
    private static final String JSONKEY_GW = "Gateway";
    
    public DhcpSettings(boolean useDhcp, String staticIP, String subnetMask, String defaultGateway)
                    throws FormatException {
        this(useDhcp, staticIP, Utils.netmaskStringToInt(subnetMask), defaultGateway);
    }

    public DhcpSettings(boolean useDhcp, String staticIP, int subnetMask, String defaultGatweay)
                    throws FormatException {

        this.useDhcp = useDhcp;

        if (!useDhcp) {

            if (!validateParams(staticIP, defaultGatweay, subnetMask)) {
                throw new FormatException("Invalid dhcp parameters!");
            }

            this._staticIP = staticIP;
            this._mask = subnetMask;
            this._defGw = defaultGatweay;

        }

    }

    private DhcpSettings(){
        // inner use
    }
    
    public static DhcpSettings parseSavedSettings(String staticIPwithMask, String defaultGateway){
        
        try {

            String[] ipm = staticIPwithMask.split("/");
            String ip = ipm[0];
            int mask = Integer.parseInt(ipm[1]);

            return new DhcpSettings(false, ip, mask, defaultGateway);
            

        } catch (Exception e) {
            Utils.logError("Exception while parsing DhcpSettings for saved network. Reverting to dhcp.", e);
            return null;
        }
        
    }
    
    public static DhcpSettings getDefault(){
        try {
            return new DhcpSettings(true, null, 24, null);
        } catch (FormatException e) {
            //no format exception can happen.
            return null;
        }
    }
    
    private boolean validateParams(String ip, String gateway, int mask) {

        if (isValidAddress(ip) && isValidAddress(gateway) && isValidMask(mask)) {
            return true;
        } else {
            return false;
        }

    }
    
    public static boolean isValidAddress(String ipAddress){
        return InetAddressUtils.isIPv4Address(ipAddress);
    }
    
    public static boolean isValidMaks(String mask){
        int m = Utils.netmaskStringToInt(mask);
        return isValidMask(m);
    }

    public static boolean isValidMask(int mask){
        if (mask >= 8 && mask <= 32){
            return true;
        }else{
            return false;
        }
    }
    
    public String getStaticIP() {
        if (_staticIP == null){
            return "";
        }
        return _staticIP;
    }

    public String getStaticIPwithMask() {
        return getStaticIP() + "/" + String.valueOf(getSubnetMaskInt());
    }

    public int getSubnetMaskInt() {
        return _mask;
    }
    
    public String getSubnetMaskString(){
        String v = Utils.netmaskIntToString(_mask);
        if (v == null){
            return "";
        }
        return v;
    }
    
    public String getDefaultGateway() {
        if (_defGw == null){
            return "";
        }
        return _defGw;
    }

    public JSONObject toJson(){
        
        JSONObject j = new JSONObject();
        
        try {
            j.put(JSONKEY_DHCP, useDhcp);
            j.put(JSONKEY_GW, getDefaultGateway());
            j.put(JSONKEY_STATIC_IP, getStaticIPwithMask());
            
            return j;
            
        } catch (JSONException e) {
            Utils.logError("Exception while converting DhcpSettings to JSON.", e);
            return null;
        }
        
        
    }
    
    public static DhcpSettings fromJsonObject(JSONObject json){
        
        if (json == null){
            return null;
        }
        
        DhcpSettings sets = new DhcpSettings();
        
        try {
            
            sets.useDhcp = json.getBoolean(JSONKEY_DHCP);
            
            if (json.has(JSONKEY_GW) && ! json.isNull(JSONKEY_GW)){
                sets._defGw = json.getString(JSONKEY_GW);
            }
            
            if (json.has(JSONKEY_STATIC_IP) && !json.isNull(JSONKEY_STATIC_IP)){
                
                String[] splt = json.getString(JSONKEY_STATIC_IP).split("/");
                sets._staticIP = splt[0];
                sets._mask = Integer.parseInt(splt[1]);
                
            }
            
            return sets;
            
        } catch (Exception e) {
            Utils.logError("Exception while parsing json object to DhcpSettings", e);
            return null;
        }
        
    }
    
}
