//
// Copyright 2017 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.activities;

import fil.libre.repwifiapp.ActivityLauncher;
import fil.libre.repwifiapp.Commons;
import fil.libre.repwifiapp.R;
import fil.libre.repwifiapp.ActivityLauncher.RequestCode;
import fil.libre.repwifiapp.helpers.AccessPointInfo;
import fil.libre.repwifiapp.helpers.ConnectionStatus;
import fil.libre.repwifiapp.helpers.OpenVpnManager;
import fil.libre.repwifiapp.helpers.Utils;
import android.content.Intent;
import android.os.Bundle;
import android.os.RemoteException;
import android.view.View;
import android.widget.Button;
import android.widget.TextView;
import android.widget.Toast;

public class ShowStatusActivity extends MenuEnabledActivity {

    private ConnectionStatus status;

    // private AccessPointInfo info;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_show_status);

        if (getIntent().hasExtra(ActivityLauncher.EXTRA_CONSTATUS)) {
            this.status = (ConnectionStatus) getIntent().getSerializableExtra(
                            ActivityLauncher.EXTRA_CONSTATUS);
        }

        try {
            showStatus(false);
        } catch (Exception e) {
            Utils.logError("Exception on showStatus", e);
        }

    }

    @Override
    public void onRestart() {
        super.onRestart();
        try {
            showStatus(true);
        } catch (Exception e) {
            Utils.logError("Exception on showStatus", e);
        }
    }
    
    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent intent) {

        switch (requestCode) {

        case RequestCode.VPN_PERMISSION:
            
            if (resultCode == RESULT_OK) {
                endLaunchVpn(true);
            } else {
                endLaunchVpn(false);
            }

            break;

        default:

            break;

        }

    }

    private void setMessage(String msg) {
        TextView view = (TextView) findViewById(R.id.txt_status);
        view.setText(msg);
    }

    private void showStatus(boolean refresh) throws Exception {

        if (refresh || status == null) {
            this.status = Commons.connectionEngine.getConnectionStatus();
        }

        if (status == null) {
            this.finish();

        } else if (this.status.isConnected()) {
            Utils.logDebug("StatusActivity isConnected,showing buttons");
            setMessage(getString(R.string.msg_connected_to) + " " + status.SSID + "\n\n"
                            + getString(R.string.text_ip_address) + ": " + status.IP + "\n");
            toggleBtnDisconnect(true);
            beginLauncVpn();
            
        } else {
            Utils.logDebug("StatusActivity status Else");
            setMessage(getString(R.string.text_status) + ":\n" + status.status);
            toggleBtnDisconnect(false);
        }

        Commons.updateNotification(this);

    }

    private void toggleBtnDisconnect(boolean enable) {

        Button b = (Button) findViewById(R.id.btn_disconnect);
        Button bk = (Button) findViewById(R.id.btn_back);
        b.setEnabled(enable);
        bk.setEnabled(!enable);

        if (enable) {
            b.setVisibility(View.VISIBLE);
            bk.setVisibility(View.INVISIBLE);
        } else {
            b.setVisibility(View.INVISIBLE);
            bk.setVisibility(View.VISIBLE);
        }

    }

    public void onBtnDisconnectClick(View v) {

        disconnectVpn();
        
        boolean res = Commons.connectionEngine.disconnect();
        String msg = "";
        if (res) {
            msg = getString(R.string.msg_disconnected);
        } else {
            msg = getString(R.string.msg_disconnect_fail);
        }

        Toast toast = Toast.makeText(getApplicationContext(), msg, Toast.LENGTH_SHORT);
        toast.show();

        try {
            showStatus(true);
        } catch (Exception e) {
            Utils.logError("Exception on showStatus", e);
        }

    }

    private void disconnectVpn() {

        if (! OpenVpnManager.isVpnConnected()){
            return;
        }
        
        if (OpenVpnManager.disconnect()) {
            Toast t = Toast.makeText(this, getString(R.string.msg_vpn_disconnect),
                            Toast.LENGTH_LONG);
            t.show();

        } else {
            Commons.showMessage(getString(R.string.msg_vpn_disconnect_error), this);
        }

    }

    private void beginLauncVpn() throws Exception {

        if (OpenVpnManager.isVpnConnected()){
            // already connected;
            return;
        }
        
        if (!OpenVpnManager.isExternalAppInstalled(this)) {
            Utils.logDebug("External VPN app not installed.");
            return;
        }
        
        if (getVpnNameIfAny() == null){
            Utils.logDebug("No vpn profile set. Exiting beginLaunchVpn()");
            return;
        }

        // first, we make sure we have permission to use the vpn service.
        Intent pi;
        
        try {
            pi = OpenVpnManager.askApiPermissionsGetIntent();
        } catch (RemoteException e) {
            Utils.logError("Exception while asking for VPN permission", e);
            Toast t = Toast.makeText(getApplicationContext(), getString(R.string.msg_vpn_connect_error), Toast.LENGTH_LONG);
            t.show();
            return;   
            
        }
        
        if (pi == null){
            // no need to ask for permission
            Utils.logDebug("No need for vpn permission: going to endLaunchVpn.");
            endLaunchVpn(true);
            
        } else{
            // launch the intent to ask permission
            Utils.logDebug("Need to ask for vpn permission. Starting intent..");
            startActivityForResult(pi, ActivityLauncher.RequestCode.VPN_PERMISSION);
        }

    }

    private void endLaunchVpn(boolean permissionGranted) {

        try {
           
           if (!permissionGranted) {
               // warn user that permission must be granted
               Utils.logDebug("User rejected vpn permission.");
               String msg = getString(R.string.msg_vpn_no_permission).replace(
                               OpenVpnManager.PLACEHOLDER_APPNAME, OpenVpnManager.APP_COMMON_NAME);
               Commons.showMessage(msg, this);
               return;
           }
           
           String profname = getVpnNameIfAny();
           
           // check if profile exists
           String profUuid = OpenVpnManager.getUuidFromName(profname);
           if (profUuid == null){
               // warn user that selected profile doesn't exist
               Commons.showMessage(getString(R.string.msg_vpn_wrong_profile), this);
               return;
           }
                      
           if (OpenVpnManager.startVpn(profUuid)){
               Toast t = Toast.makeText(this, getString(R.string.msg_vpn_launched), Toast.LENGTH_LONG);
               t.show();            
           } else {
               Commons.showMessage(getString(R.string.msg_vpn_connect_error),this);
           }
           
       } catch (Exception e) {
           Utils.logError("Exception while endLaunchVpn", e);
           
       }

    }

    private String getVpnNameIfAny(){
        
        if (status == null) {
            return null;
        }

        AccessPointInfo i = status.getNetworkDetails();
        if (i == null) {
            return null;
        }

        String profname = i.getVpnProfileName();
        if (profname == null || profname.isEmpty()) {
            return null;
        } else {
            return profname;
        }
         
    }
    
    public void onBtnMainClick(View v) {
        finish();
    }

    @Override
    public void onBackPressed() {
        moveTaskToBack(true);
    }
    
}
