/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.tasting;

import android.content.ContentValues;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;

import de.retujo.bierverkostung.beer.Beer;
import de.retujo.bierverkostung.common.CommaDelimiterTrimmer;
import de.retujo.bierverkostung.data.BierverkostungContract.TastingEntry;
import de.retujo.bierverkostung.data.ParcelUnwrapper;
import de.retujo.bierverkostung.data.ParcelWrapper;
import de.retujo.bierverkostung.data.UriToIdFunction;
import de.retujo.java.util.Maybe;
import de.retujo.java.util.ObjectUtil;

import static android.text.TextUtils.isEmpty;
import static de.retujo.java.util.Conditions.argumentNotEmpty;
import static de.retujo.java.util.Conditions.isNotNull;
import static de.retujo.java.util.ObjectUtil.areEqual;

/**
 * A mutable builder with a fluent API for creating immutable instances of {@link Tasting}.
 *
 * @since 1.0.0
 */
@NotThreadSafe
final class TastingBuilder {

    private final String date;
    private final Beer beer;
    private long id;
    private String location;
    private OpticalAppearance opticalAppearance;
    private Scent scent;
    private Taste taste;

    private String foodRecommendation;
    private String totalImpressionDescription;
    private int totalImpressionRating;

    private TastingBuilder(final String theDate, final Beer theBeer) {
        date = theDate;
        beer = theBeer;
        id = UriToIdFunction.NO_ID;
        opticalAppearance = OpticalAppearanceBuilder.getInstance().build();
        scent = ScentBuilder.getInstance().build();
        taste = TasteBuilder.getInstance().build();
    }

    /**
     * Returns a new instance of {@code TastingBuilder}.
     *
     * @param date the date of the tasting in ISO 8601 format.
     * @param beer the tasted beer.
     * @return the instance.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if {@code date} is empty.
     */
    public static TastingBuilder newInstance(final String date, final Beer beer) {
        return new TastingBuilder(argumentNotEmpty(date, "date"), isNotNull(beer, "beer"));
    }

    public TastingBuilder id(final long id) {
        if (0 < id) {
            this.id = id;
        }
        return this;
    }

    public TastingBuilder location(final CharSequence location) {
        if (!isEmpty(location)) {
            this.location = location.toString();
        }
        return this;
    }

    public TastingBuilder opticalAppearance(@Nullable final OpticalAppearance opticalAppearance) {
        if (null != opticalAppearance) {
            this.opticalAppearance = opticalAppearance;
        }
        return this;
    }

    @Nonnull
    public TastingBuilder scent(@Nullable final Scent scent) {
        if (null != scent) {
            this.scent = scent;
        }
        return this;
    }

    @Nonnull
    public TastingBuilder taste(@Nullable final Taste taste) {
        if (null != taste) {
            this.taste = taste;
        }
        return this;
    }

    @Nonnull
    public TastingBuilder foodRecommendation(@Nullable final CharSequence foodRecommendation) {
        if (!isEmpty(foodRecommendation)) {
            this.foodRecommendation = CommaDelimiterTrimmer.trim(foodRecommendation);
        }
        return this;
    }

    @Nonnull
    public TastingBuilder totalImpressionRating(final int rating) {
        totalImpressionRating = rating;
        return this;
    }

    @Nonnull
    public TastingBuilder totalImpressionDescription(@Nullable final CharSequence description) {
        if (!isEmpty(description)) {
            totalImpressionDescription = description.toString();
        }
        return this;
    }

    @Nonnull
    public Tasting build() {
        return new ImmutableTasting(this);
    }

    /**
     * Immutable implementation of {@link Tasting}.
     *
     * @since 1.0.0
     */
    @Immutable
    static final class ImmutableTasting implements Tasting {
        /**
         * Creator which creates instances of {@code ImmutableTasting} from a Parcel.
         */
        public static final Parcelable.Creator<Tasting> CREATOR = new ImmutableTastingCreator();

        private static final byte MAX_CONTENT_VALUES = 20;

        private final long id;
        private final String date;
        private final String location;
        private final Beer beer;
        private final OpticalAppearance opticalAppearance;
        private final Scent scent;
        private final Taste taste;
        private final String foodRecommendation;
        private final String totalImpressionDescription;
        private final int totalImpressionRating;

        private ImmutableTasting(final TastingBuilder builder) {
            id = builder.id;
            date = builder.date;
            location = builder.location;
            beer = builder.beer;
            opticalAppearance = builder.opticalAppearance;
            scent = builder.scent;
            taste = builder.taste;
            foodRecommendation = builder.foodRecommendation;
            totalImpressionDescription = builder.totalImpressionDescription;
            totalImpressionRating = builder.totalImpressionRating;
        }

        @Nonnull
        @Override
        public Maybe<Long> getId() {
            return Maybe.of(id);
        }

        @Nonnull
        @Override
        public String getDate() {
            return date;
        }

        @Nonnull
        @Override
        public Maybe<String> getLocation() {
            return Maybe.ofNullable(location);
        }

        @Nonnull
        @Override
        public Beer getBeer() {
            return beer;
        }

        @Nonnull
        @Override
        public OpticalAppearance getOpticalAppearance() {
            return opticalAppearance;
        }

        @Nonnull
        @Override
        public Scent getScent() {
            return scent;
        }

        @Nonnull
        @Override
        public Taste getTaste() {
            return taste;
        }

        @Nonnull
        @Override
        public Maybe<String> getFoodRecommendation() {
            return Maybe.ofNullable(foodRecommendation);
        }

        @Override
        public Maybe<String> getTotalImpressionDescription() {
            return Maybe.ofNullable(totalImpressionDescription);
        }

        @Override
        public int getTotalImpressionRating() {
            return totalImpressionRating;
        }

        @Nonnull
        @Override
        public Uri getContentUri() {
            return TastingEntry.CONTENT_URI;
        }

        @Nonnull
        @Override
        public ContentValues asContentValues() {
            final ContentValues result = new ContentValues(MAX_CONTENT_VALUES);
            result.put(TastingEntry.COLUMN_DATE.toString(), date);
            if (UriToIdFunction.NO_ID != id) {
                result.put(TastingEntry.COLUMN_ID.toString(), id);
            }
            putIfNotEmpty(TastingEntry.COLUMN_LOCATION, location, result);
            result.put(TastingEntry.COLUMN_BEER_ID.toString(), beer.getId().orElse(UriToIdFunction.NO_ID));
            if (null != opticalAppearance) {
                result.putAll(opticalAppearance.asContentValues());
            }
            if (null != scent) {
                result.putAll(scent.asContentValues());
            }
            if (null != taste) {
                result.putAll(taste.asContentValues());
            }
            putIfNotEmpty(TastingEntry.COLUMN_FOOD_RECOMMENDATION, foodRecommendation, result);
            putIfNotEmpty(TastingEntry.COLUMN_TOTAL_IMPRESSION_DESCRIPTION, totalImpressionDescription, result);
            result.put(TastingEntry.COLUMN_TOTAL_IMPRESSION_RATING.toString(), totalImpressionRating);
            return result;
        }

        private static void putIfNotEmpty(final CharSequence key, final String value,
                final ContentValues contentValues) {
            if (!isEmpty(value)) {
                contentValues.put(key.toString(), value);
            }
        }

        @Override
        public int describeContents() {
            return 0;
        }

        @SuppressWarnings({"squid:S1067", "squid:S3776"})
        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            final ImmutableTasting that = (ImmutableTasting) o;

            return id == that.id
                    && areEqual(date, that.date)
                    && areEqual(location, that.location)
                    && areEqual(beer, that.beer)
                    && areEqual(opticalAppearance, that.opticalAppearance)
                    && areEqual(scent, that.scent)
                    && areEqual(taste, that.taste)
                    && areEqual(foodRecommendation, that.foodRecommendation)
                    && areEqual(totalImpressionDescription, that.totalImpressionDescription)
                    && totalImpressionRating == that.totalImpressionRating;
        }

        @Override
        public int hashCode() {
            return ObjectUtil.hashCode(id, date, location, beer, opticalAppearance, scent, taste, foodRecommendation,
                    totalImpressionDescription, totalImpressionRating);
        }

        @Override
        public void writeToParcel(final Parcel dest, final int flags) {
            ParcelWrapper.of(dest)
                    .wrap(id)
                    .wrap(date)
                    .wrap(beer)
                    .wrap(location)
                    .wrap(opticalAppearance)
                    .wrap(scent)
                    .wrap(taste)
                    .wrap(foodRecommendation)
                    .wrap(totalImpressionDescription)
                    .wrap(totalImpressionRating);
        }
    }

    /**
     * This class creates instances of {@code ImmutableTasting} from a Parcel.
     *
     * @since 1.0.0
     */
    @Immutable
    private static final class ImmutableTastingCreator implements Parcelable.Creator<Tasting> {
        @Override
        public Tasting createFromParcel(final Parcel source) {
            final ParcelUnwrapper parcelUnwrapper = ParcelUnwrapper.of(source);
            final long readId = parcelUnwrapper.unwrapLong();
            final String readDate = parcelUnwrapper.unwrapString();
            final Beer readBeer = parcelUnwrapper.unwrapParcelable();

            return TastingBuilder.newInstance(readDate, readBeer)
                    .id(readId)
                    .location(parcelUnwrapper.unwrapString())
                    .opticalAppearance(parcelUnwrapper.unwrapParcelable())
                    .scent(parcelUnwrapper.unwrapParcelable())
                    .taste(parcelUnwrapper.unwrapParcelable())
                    .foodRecommendation(parcelUnwrapper.unwrapString())
                    .totalImpressionDescription(parcelUnwrapper.unwrapString())
                    .totalImpressionRating(parcelUnwrapper.unwrapInt())
                    .build();
        }

        @Override
        public Tasting[] newArray(final int size) {
            return new Tasting[size];
        }
    }

}
