/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import android.text.TextUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;

import static de.retujo.java.util.Conditions.argumentNotEmpty;
import static de.retujo.java.util.Conditions.isNotNull;

/**
 * A very basic Java DSL for SQL queries. This class could be a lot more generic and actually aware of the SQL
 * semantics but for the moment it is sufficient.
 *
 * @since 1.0.0
 */
@NotThreadSafe
final class QuerySqlBuilder {

    private final StringBuilder stringBuilder;

    private QuerySqlBuilder(final StringBuilder theStringBuilder) {
        stringBuilder = theStringBuilder;
    }

    /**
     * Returns a new instance of {@code QuerySqlBuilder}.
     *
     * @param column the column of the table column to be included in the query result.
     * @param furtherColumns further table columns to be included in the query result.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if {@code column} is empty.
     */
    @Nonnull
    public static QuerySqlBuilder select(@Nonnull final Column column,
            @Nonnull final Column... furtherColumns) {
        argumentNotEmpty(column, "column");
        checkFurtherColumns(furtherColumns);
        final List<Column> allColumns = new ArrayList<>();
        allColumns.add(column);
        Collections.addAll(allColumns, furtherColumns);
        return select(allColumns);
    }

    private static void checkFurtherColumns(@Nonnull final Object o) {
        isNotNull(o, "further columns");
    }

    /**
     * Returns a new instance of {@code QuerySqlBuilder}.
     *
     * @param columns the columns to be included in the query result.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if {@code column} is empty.
     */
    @Nonnull
    public static QuerySqlBuilder select(@Nonnull final Collection<Column> columns) {
        isNotNull(columns, "columns");
        final StringBuilder sb = new StringBuilder("SELECT ");
        final String commaDelimiter = ", ";
        String delimiter = "";
        for (final Column column : columns) {
            sb.append(delimiter).append(column.getQualifiedName()).append(" AS ").append(column.getAlias());
            delimiter = commaDelimiter;
        }

        return new QuerySqlBuilder(sb);
    }

    /**
     * Returns a new instance of {@code QuerySqlBuilder}.
     *
     * @param column the column of the table column to be included in the query result.
     * @param furtherColumns further table columns to be included in the query result.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if {@code column} is empty.
     */
    @Nonnull
    public static QuerySqlBuilder selectDistinct(@Nonnull final Column column,
            @Nonnull final Column... furtherColumns) {
        argumentNotEmpty(column, "column");
        checkFurtherColumns(furtherColumns);
        final StringBuilder sb = new StringBuilder("SELECT DISTINCT ");
        sb.append(column.getQualifiedName()).append(" AS ").append(column.getAlias());
        for (final Column furtherColumn : furtherColumns) {
            sb.append(", ").append(furtherColumn.getQualifiedName()).append(" AS ").append(furtherColumn.getAlias());
        }

        return new QuerySqlBuilder(sb);
    }

    /**
     * Returns a new instance of {@code QuerySqlBuilder}.
     *
     * @param columnName the name of the table column to be included in the query result.
     * @param furtherColumnNames names of further table columns to be included in the query result.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if {@code columnName} is empty.
     */
    @Nonnull
    public static QuerySqlBuilder selectDistinct(@Nonnull final CharSequence columnName,
            @Nonnull final CharSequence... furtherColumnNames) {
        argumentNotEmpty(columnName, "columnName");
        checkFurtherColumns(furtherColumnNames);
        final StringBuilder sb = new StringBuilder("SELECT DISTINCT ");
        sb.append(columnName);
        for (final CharSequence furtherColumnName : furtherColumnNames) {
            sb.append(", ").append(furtherColumnName);
        }

        return new QuerySqlBuilder(sb);
    }


    /**
     * Temporarily renames the table or column to the specified alias. The table or column to be renamed has has to be
     * specified before calling this method by invoking {@link #select(Column, Column...)} for example.
     *
     * @param alias the new temporary name.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if {@code alias} is {@code null}.
     * @throws IllegalArgumentException if {@code alias} is empty.
     */
    @Nonnull
    public QuerySqlBuilder as(@Nonnull final CharSequence alias) {
        argumentNotEmpty(alias, "new column name");
        stringBuilder.append(" AS ").append(alias);
        return this;
    }

    /**
     * Sets the name of the table from which data should be retrieved.
     *
     * @param tableName the name of the table from which data should be retrieved.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if {@code tableName} is {@code null}.
     * @throws IllegalArgumentException if {@code tableName} is empty.
     */
    @Nonnull
    public QuerySqlBuilder from(@Nonnull final CharSequence tableName) {
        argumentNotEmpty(tableName, "table name");
        stringBuilder.append(" FROM ").append(tableName);
        return this;
    }

    /**
     * Sets the name of the table which should be joined with the source table.
     *
     * @param tableName name of the table which should be joined with the source table.
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if {@code tableName} is {@code null}.
     * @throws IllegalArgumentException if {@code tableName} is empty.
     */
    @Nonnull
    public QuerySqlBuilder leftOuterJoin(@Nonnull final CharSequence tableName) {
        argumentNotEmpty(tableName, "table name");
        stringBuilder.append(" LEFT OUTER JOIN ").append(tableName);
        return this;
    }

    /**
     * Sets the column names whose content should match for joining data.
     *
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if any argument is empty.
     */
    @Nonnull
    public QuerySqlBuilder on(@Nonnull final CharSequence firstMatchingColumnName,
            @Nonnull final CharSequence secondMatchingColumnName) {
        argumentNotEmpty(firstMatchingColumnName, "first matching column name");
        argumentNotEmpty(secondMatchingColumnName, "second matching column name");
        stringBuilder.append(" ON ").append(firstMatchingColumnName).append('=').append(secondMatchingColumnName);
        return this;
    }

    /**
     * Sets the columns whose content should match for joining data.
     *
     * @return this builder instance to allow Method Chaining.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if any argument is empty.
     */
    @Nonnull
    public QuerySqlBuilder on(@Nonnull final Column firstMatchingColumn,
            @Nonnull final Column secondMatchingColumn) {
        return on(firstMatchingColumn.getQualifiedName(), secondMatchingColumn.getQualifiedName());
    }

    /**
     * Sets the sort order.
     *
     * @param sortOrder the sort order.
     * @return this builder instance to allow Method Chaining.
     */
    @Nonnull
    public QuerySqlBuilder orderBy(@Nullable final String sortOrder) {
        if (!TextUtils.isEmpty(sortOrder)) {
            stringBuilder.append(" ORDER BY ").append(sortOrder);
        }
        return this;
    }

    /**
     * Returns a fully-fledged SQL string for querying data.
     *
     * @return the SQL string.
     */
    @Nonnull
    @Override
    public String toString() {
        stringBuilder.append(';');
        return stringBuilder.toString();
    }

}
