/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import android.content.Context;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import javax.annotation.concurrent.NotThreadSafe;

import de.retujo.bierverkostung.data.BierverkostungContract.BeerEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.BeerStyleEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.BreweryEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.CountryEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.TastingEntry;

import static de.retujo.java.util.Conditions.isNotNull;

/**
 * A helper class to manage creation and version management of the application's database.
 *
 * @since 1.0.0
 */
@NotThreadSafe
final class BierverkostungDbHelper extends SQLiteOpenHelper {

    private static final String TAG = BierverkostungDbHelper.class.getName();
    private static final String DATABASE_NAME = "bierverkostung.db";
    private static final int VERSION = 1;
    private static final String NOT_NULL_CONSTRAINT = "NOT NULL";

    /**
     * Constructs a new {@code BierverkostungDbHelper} object.
     *
     * @param context the Context to use to open or create the database.
     * @throws NullPointerException if {@code context} is {@code null}.
     */
    public BierverkostungDbHelper(final Context context) {
        super(isNotNull(context, "The context must not be null!"), DATABASE_NAME, null, VERSION);
    }

    @Override
    public void onCreate(final SQLiteDatabase db) {
        createCountriesTable(db);
        createBreweriesTable(db);
        createBeerStylesTable(db);
        createBeersTable(db);
        createTastingsTable(db);
    }

    private static void createCountriesTable(final SQLiteDatabase db) {
        final String sql = CreateTableSqlBuilder.newInstance(CountryEntry.TABLE)
                .withPrimaryKeyColumn(CountryEntry.COLUMN_ID)
                .withColumn(CountryEntry.COLUMN_NAME, NOT_NULL_CONSTRAINT)
                .toString();

        db.execSQL(sql);
    }

    private static void createBreweriesTable(final SQLiteDatabase db) {
        final String sql = CreateTableSqlBuilder.newInstance(BreweryEntry.TABLE)
                .withPrimaryKeyColumn(BreweryEntry.COLUMN_ID)
                .withColumn(BreweryEntry.COLUMN_NAME, NOT_NULL_CONSTRAINT)
                .withColumn(BreweryEntry.COLUMN_LOCATION)
                .withColumn(BreweryEntry.COLUMN_COUNTRY_ID)
                .withForeignKey(BreweryEntry.COLUMN_COUNTRY_ID, CountryEntry.COLUMN_ID)
                .toString();

        db.execSQL(sql);
    }

    private static void createBeerStylesTable(final SQLiteDatabase db) {
        final String sql = CreateTableSqlBuilder.newInstance(BeerStyleEntry.TABLE)
                .withPrimaryKeyColumn(BeerStyleEntry.COLUMN_ID)
                .withColumn(BeerStyleEntry.COLUMN_NAME, NOT_NULL_CONSTRAINT)
                .toString();

        db.execSQL(sql);
    }

    private static void createBeersTable(final SQLiteDatabase db) {
        final String sql = CreateTableSqlBuilder.newInstance(BeerEntry.TABLE)
                .withPrimaryKeyColumn(BeerEntry.COLUMN_ID)
                .withColumn(BeerEntry.COLUMN_NAME)
                .withColumn(BeerEntry.COLUMN_BREWERY_ID)
                .withColumn(BeerEntry.COLUMN_STYLE_ID)
                .withColumn(BeerEntry.COLUMN_ORIGINAL_WORT)
                .withColumn(BeerEntry.COLUMN_ALCOHOL)
                .withColumn(BeerEntry.COLUMN_IBU)
                .withColumn(BeerEntry.COLUMN_INGREDIENTS)
                .withColumn(BeerEntry.COLUMN_SPECIFICS)
                .withColumn(BeerEntry.COLUMN_NOTES)
                .withForeignKey(BeerEntry.COLUMN_STYLE_ID, BeerStyleEntry.COLUMN_ID)
                .withForeignKey(BeerEntry.COLUMN_BREWERY_ID, BreweryEntry.COLUMN_ID)
                .toString();

        db.execSQL(sql);
    }

    private static void createTastingsTable(final SQLiteDatabase db) {
        final String sql = CreateTableSqlBuilder.newInstance(TastingEntry.TABLE)
                .withPrimaryKeyColumn(TastingEntry.COLUMN_ID)
                .withColumn(TastingEntry.COLUMN_DATE)
                .withColumn(TastingEntry.COLUMN_LOCATION)
                .withColumn(TastingEntry.COLUMN_BEER_ID)
                .withColumn(TastingEntry.COLUMN_BEER_COLOUR)
                .withColumn(TastingEntry.COLUMN_BEER_COLOUR_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_COLOUR_EBC)
                .withColumn(TastingEntry.COLUMN_CLARITY_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_FOAM_COLOUR)
                .withColumn(TastingEntry.COLUMN_FOAM_STRUCTURE)
                .withColumn(TastingEntry.COLUMN_FOAM_STABILITY)
                .withColumn(TastingEntry.COLUMN_FRUIT_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_FRUIT_RATING)
                .withColumn(TastingEntry.COLUMN_FLOWER_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_FLOWER_RATING)
                .withColumn(TastingEntry.COLUMN_VEGETAL_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_VEGETAL_RATING)
                .withColumn(TastingEntry.COLUMN_SPICY_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_SPICY_RATING)
                .withColumn(TastingEntry.COLUMN_WARMTH_MINTED_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_WARMTH_MINTED_RATING)
                .withColumn(TastingEntry.COLUMN_BIOLOGICAL_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_BIOLOGICAL_RATING)
                .withColumn(TastingEntry.COLUMN_BITTERNESS_RATING)
                .withColumn(TastingEntry.COLUMN_SWEETNESS_RATING)
                .withColumn(TastingEntry.COLUMN_ACIDITY_RATING)
                .withColumn(TastingEntry.COLUMN_MOUTHFEEL_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_FULL_BODIED_RATING)
                .withColumn(TastingEntry.COLUMN_BODY_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_AFTERTASTE_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_AFTERTASTE_RATING)
                .withColumn(TastingEntry.COLUMN_FOOD_RECOMMENDATION)
                .withColumn(TastingEntry.COLUMN_TOTAL_IMPRESSION_DESCRIPTION)
                .withColumn(TastingEntry.COLUMN_TOTAL_IMPRESSION_RATING)
                .withForeignKey(TastingEntry.COLUMN_BEER_ID, BeerEntry.COLUMN_ID)
                .toString();

        db.execSQL(sql);
    }

    @Override
    public void onUpgrade(final SQLiteDatabase db, final int oldVersion, final int newVersion) {
        throw new UnsupportedOperationException("Not yet implemented.");
    }

}
