/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import android.content.ContentProvider;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.net.Uri;
import android.support.annotation.Nullable;

import java.text.MessageFormat;
import java.util.List;

import javax.annotation.Nonnull;

import de.retujo.bierverkostung.data.BierverkostungContract.BeerEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.BeerStyleEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.BreweryEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.CountryEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.TastingEntry;
import de.retujo.java.util.Maybe;

import static de.retujo.java.util.Conditions.isNotNull;

/**
 * This class provides content to the application by communicating with the database.
 *
 * @since 1.0.0
 */
public final class BierverkostungContentProvider extends ContentProvider {

    private static final String TAG = BierverkostungContentProvider.class.getName();

    private SQLiteOpenHelper dbHelper;

    /**
     * Constructs a new {@code BierverkostungContentProvider} object.
     */
    public BierverkostungContentProvider() {
        dbHelper = null;
    }

    @Override
    public boolean onCreate() {
        dbHelper = new BierverkostungDbHelper(getContext());
        return true;
    }

    @Nullable
    @Override
    public Cursor query(@Nonnull final Uri uri, final String[] projection, final String selection,
            final String[] selectionArgs, final String sortOrder) {
        final SQLiteDatabase db = dbHelper.getReadableDatabase();
        final BierverkostungUriMatcher uriMatcher = BierverkostungUriMatcher.getInstance();
        final Cursor result;
        switch (uriMatcher.match(uri)) {
            case BEERS:
                result = queryAllBeers(db, sortOrder);
                break;
            case BEER_STYLES:
                result = queryAllBeerStyles(db, sortOrder);
                break;
            case BREWERIES:
                result = queryAllBreweries(db, sortOrder);
                break;
            case COUNTRIES:
                result = queryAllCountries(db, sortOrder);
                break;
            case SINGLE_COLUMN_VALUES:
                final String tableName = getTableName(uri);
                final String columnName = getFirstColumnName(projection);
                result = queryDistinctColumnValues(db, tableName, columnName);
                break;
            case TASTINGS:
                result = queryAllTastings(db, sortOrder);
                break;
            default:
                throw new UnsupportedOperationException(MessageFormat.format("Unknown URI <{0}> for query!", uri));
        }

        final Context context = getContext();
        if (null != result && null != context) {
            final ContentResolver contentResolver = context.getContentResolver();
            result.setNotificationUri(contentResolver, uri);
        }

        return result;
    }

    private static Cursor queryAllBeers(final SQLiteDatabase db, final String sortOrder) {
        final List<Column> queryColumns = joinAllColumns(BeerEntry.TABLE, BreweryEntry.TABLE, BeerStyleEntry.TABLE,
                CountryEntry.TABLE);

        final String query = QuerySqlBuilder.select(queryColumns)
                .from(BeerEntry.TABLE)
                .leftOuterJoin(BreweryEntry.TABLE).on(BeerEntry.COLUMN_BREWERY_ID, BreweryEntry.COLUMN_ID)
                .leftOuterJoin(CountryEntry.TABLE).on(BreweryEntry.COLUMN_COUNTRY_ID, CountryEntry.COLUMN_ID)
                .leftOuterJoin(BeerStyleEntry.TABLE).on(BeerEntry.COLUMN_STYLE_ID, BeerStyleEntry.COLUMN_ID)
                .orderBy(sortOrder)
                .toString();

        return db.rawQuery(query, null);
    }

    private static List<Column> joinAllColumns(final Table firstTable, final Table secondTable,
            final Table ... furtherTables) {
        final List<Column> result = firstTable.getColumns();
        result.addAll(secondTable.getColumns());
        for (final Table furtherTable : furtherTables) {
            result.addAll(furtherTable.getColumns());
        }
        return result;
    }

    private static Cursor queryAllBeerStyles(final SQLiteDatabase db, final String sortOrder) {
        final Table countryTable = BeerStyleEntry.TABLE;
        final String query = QuerySqlBuilder.select(countryTable.getColumns())
                .from(countryTable)
                .orderBy(sortOrder)
                .toString();

        return db.rawQuery(query, null);
    }

    private static Cursor queryAllBreweries(final SQLiteDatabase db, final String sortOrder) {
        final List<Column> queryColumns = joinAllColumns(BreweryEntry.TABLE, CountryEntry.TABLE);

        final String query = QuerySqlBuilder.select(queryColumns)
                .from(BreweryEntry.TABLE)
                .leftOuterJoin(CountryEntry.TABLE).on(BreweryEntry.COLUMN_COUNTRY_ID, CountryEntry.COLUMN_ID)
                .orderBy(sortOrder)
                .toString();

        return db.rawQuery(query, null);
    }

    private static Cursor queryAllCountries(final SQLiteDatabase db, final String sortOrder) {
        final Table countryTable = CountryEntry.TABLE;
        final String query = QuerySqlBuilder.select(countryTable.getColumns())
                .from(countryTable)
                .orderBy(sortOrder)
                .toString();

        return db.rawQuery(query, null);
    }

    private static String getTableName(final Uri contentUri) {
        final List<String> pathSegments = contentUri.getPathSegments();
        isNotNull(pathSegments, "path segments of the Content URI");
        final int minPathSegmentSize = 2;
        final int pathSegmentsSize = pathSegments.size();
        if (minPathSegmentSize > pathSegmentsSize) {
            final String pattern = "The Content URI must have at least <{0}> path segments! (<{1}>)";
            throw new IllegalArgumentException(MessageFormat.format(pattern, minPathSegmentSize, pathSegments));
        }
        // Returns the last but one path segment.
        return pathSegments.get(pathSegmentsSize - minPathSegmentSize);
    }

    private static String getFirstColumnName(final String[] columnNames) {
        isNotNull(columnNames, "column names array");
        if (0 == columnNames.length) {
            throw new IllegalArgumentException("The column names array must not be empty");
        }
        return columnNames[0];
    }

    private static Cursor queryDistinctColumnValues(final SQLiteDatabase readableDatabase, final String tableName,
            final String columnName) {

        final String query = QuerySqlBuilder.selectDistinct(columnName)
                .from(tableName)
                .toString();

        return readableDatabase.rawQuery(query, null);
    }

    private static Cursor queryAllTastings(final SQLiteDatabase db, final String sortOrder) {
        final List<Column> queryColumns = joinAllColumns(TastingEntry.TABLE, BeerEntry.TABLE, BreweryEntry.TABLE,
                CountryEntry.TABLE, BeerStyleEntry.TABLE);

        final String query = QuerySqlBuilder.select(queryColumns)
                .from(TastingEntry.TABLE)
                .leftOuterJoin(BeerEntry.TABLE).on(TastingEntry.COLUMN_BEER_ID, BeerEntry.COLUMN_ID)
                .leftOuterJoin(BreweryEntry.TABLE).on(BeerEntry.COLUMN_BREWERY_ID, BreweryEntry.COLUMN_ID)
                .leftOuterJoin(CountryEntry.TABLE).on(BreweryEntry.COLUMN_COUNTRY_ID, CountryEntry.COLUMN_ID)
                .leftOuterJoin(BeerStyleEntry.TABLE).on(BeerEntry.COLUMN_STYLE_ID, BeerStyleEntry.COLUMN_ID)
                .orderBy(sortOrder)
                .toString();

        return db.rawQuery(query, null);
    }

    @Override
    @Nullable
    public String getType(@Nonnull final Uri uri) {
        return null;
    }

    @Override
    @Nullable
    public Uri insert(@Nonnull final Uri uri, final ContentValues values) {
        final BierverkostungUriMatcher uriMatcher = BierverkostungUriMatcher.getInstance();
        switch (uriMatcher.match(uri)) {
            case BEERS:
                return insertIntoDb(BeerEntry.TABLE, values, BeerEntry.CONTENT_URI, uri);
            case BEER_STYLES:
                return insertIntoDb(BeerStyleEntry.TABLE, values, BeerStyleEntry.CONTENT_URI, uri);
            case BREWERIES:
                return insertIntoDb(BreweryEntry.TABLE, values, BreweryEntry.CONTENT_URI, uri);
            case COUNTRIES:
                return insertIntoDb(CountryEntry.TABLE, values, CountryEntry.CONTENT_URI, uri);
            case TASTINGS:
                return insertIntoDb(TastingEntry.TABLE, values, TastingEntry.CONTENT_URI, uri);
            default:
                throw new UnsupportedOperationException(MessageFormat.format("Unknown URI <{0}> for insertion!", uri));
        }
    }

    private Uri insertIntoDb(final CharSequence tableName, final ContentValues values, final Uri contentUri,
            final Uri uri) {
        final SQLiteDatabase db = dbHelper.getWritableDatabase();
        final long id = db.insertOrThrow(tableName.toString(), null, values);
        if (0 < id) {
            final Uri result = ContentUris.withAppendedId(contentUri, id);
            notifyContentResolverAboutChange(result);
            return result;
        } else {
            throw new android.database.SQLException("Failed to insert row into <" + uri + ">!");
        }
    }

    private void notifyContentResolverAboutChange(final Uri uri) {
        final Context context = getContext();
        if (null != context) {
            final ContentResolver contentResolver = context.getContentResolver();
            contentResolver.notifyChange(uri, null);
        }
    }

    @Override
    public int delete(@Nonnull final Uri uri, final String selection, final String[] selectionArgs) {
        final BierverkostungUriMatcher uriMatcher = BierverkostungUriMatcher.getInstance();
        switch (uriMatcher.match(uri)) {
            case BEER_WITH_ID:
                return deleteFromDbIfPossible(uri, BeerEntry.TABLE, TastingEntry.COLUMN_BEER_ID);
            case BEER_STYLE_WITH_ID:
                return deleteFromDb(uri, BeerStyleEntry.TABLE);
            case BREWERY_WITH_ID:
                return deleteFromDbIfPossible(uri, BreweryEntry.TABLE, BeerEntry.COLUMN_BREWERY_ID);
            case COUNTRY_WITH_ID:
                return deleteFromDb(uri, CountryEntry.TABLE);
            case TASTING_WITH_ID:
                return deleteFromDb(uri, TastingEntry.TABLE);
            default:
                throw new UnsupportedOperationException("Unknown URI <" + uri + "> for deletion!");
        }
    }

    private int deleteFromDbIfPossible(final Uri uri, final Table deleteFromTable, final Column countColumn) {
        final UriToIdFunction uriToId = UriToIdFunction.getInstance();
        final Maybe<Long> idMaybe = uriToId.apply(uri);
        if (idMaybe.isPresent()) {
            final int amountOfAfferentReferences = count(countColumn, idMaybe.get());
            if (0 == amountOfAfferentReferences) {
                return deleteFromDb(idMaybe.get(), deleteFromTable, uri);
            }
        }
        return 0;
    }

    private int count(final Column column, final long targetMatchingId) {
        final SQLiteDatabase db = dbHelper.getReadableDatabase();
        final String querySql = "SELECT COUNT(*) FROM " + column.getTable()
                + " WHERE " + column + " == " + targetMatchingId;
        final Cursor cursor = db.rawQuery(querySql, null);
        try {
            cursor.moveToFirst();
            return cursor.getInt(0);
        } finally {
            cursor.close();
        }
    }

    private int deleteFromDb(final long id, final CharSequence tableName, final Uri uri) {
        final SQLiteDatabase db = dbHelper.getWritableDatabase();
        final int result = db.delete(tableName.toString(), "_id=?", new String[]{String.valueOf(id)});
        if (0 < result) {
            notifyContentResolverAboutChange(uri);
        }
        return result;
    }

    private int deleteFromDb(final Uri uri, final CharSequence tableName) {
        int result = 0;
        final UriToIdFunction uriToId = UriToIdFunction.getInstance();
        final Maybe<Long> idMaybe = uriToId.apply(uri);
        if (idMaybe.isPresent()) {
            final String id = String.valueOf(idMaybe.get());
            final SQLiteDatabase db = dbHelper.getWritableDatabase();
            result = db.delete(tableName.toString(), "_id=?", new String[]{id});
        }

        if (0 < result) {
            notifyContentResolverAboutChange(uri);
        }
        return result;
    }

    @Override
    public int update(@Nonnull final Uri uri, final ContentValues values, final String selection,
            final String[] selectionArgs) {
        final BierverkostungUriMatcher uriMatcher = BierverkostungUriMatcher.getInstance();
        switch (uriMatcher.match(uri)) {
            case BEER_WITH_ID:
                return updateDb(uri, BeerEntry.TABLE, BeerEntry.COLUMN_ID, values);
            case BEER_STYLE_WITH_ID:
                return updateDb(uri, BeerStyleEntry.TABLE, BeerStyleEntry.COLUMN_ID, values);
            case BREWERY_WITH_ID:
                return updateDb(uri, BreweryEntry.TABLE, BreweryEntry.COLUMN_ID, values);
            case COUNTRY_WITH_ID:
                return updateDb(uri, CountryEntry.TABLE, CountryEntry.COLUMN_ID, values);
            case TASTING_WITH_ID:
                return updateDb(uri, TastingEntry.TABLE, TastingEntry.COLUMN_ID, values);
            default:
                throw new UnsupportedOperationException(MessageFormat.format("Unknown URI <{0}> for update!", uri));
        }
    }

    private int updateDb(final Uri uri, final Table table, final Column idColumn, final ContentValues values) {
        final String whereClause = idColumn + "=?";
        final String[] whereArgs = new String[]{String.valueOf(values.getAsString(idColumn.toString()))};

        final SQLiteDatabase db = dbHelper.getWritableDatabase();
        final int result = db.update(table.toString(), values, whereClause, whereArgs);

        if (0 < result) {
            notifyContentResolverAboutChange(uri);
        }
        return result;
    }

}
