/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.brewery;

import android.database.Cursor;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.country.Country;
import de.retujo.bierverkostung.country.CountryFactory;
import de.retujo.bierverkostung.data.BierverkostungContract.BreweryEntry;
import de.retujo.bierverkostung.data.CursorReader;
import de.retujo.bierverkostung.data.UriToIdFunction;

import static de.retujo.java.util.Conditions.argumentNotEmpty;

/**
 * A factory for creating {@link Brewery} instance.
 *
 * @since 1.0.0
 */
@Immutable
public final class BreweryFactory {

    private BreweryFactory() {
        throw new AssertionError();
    }

    /**
     * Returns a new immutable instance of {@link Brewery}.
     *
     * @param breweryName the breweryName of the brewery.
     * @param breweryLocation the breweryLocation of the brewery.
     * @param breweryCountry the breweryCountry of the brewery.
     * @return the instance.
     * @throws NullPointerException if {@code breweryName} is {@code null}.
     * @throws IllegalArgumentException if {@code breweryName} is empty.
     */
    @Nonnull
    public static Brewery newBrewery(@Nonnull final CharSequence breweryName,
            @Nullable final CharSequence breweryLocation, @Nullable final Country breweryCountry) {
        return newBrewery(UriToIdFunction.NO_ID, breweryName, breweryLocation, breweryCountry);
    }

    /**
     * Returns a new immutable instance of {@link Brewery}.
     *
     * @param breweryId the ID of the brewery.
     * @param breweryName the breweryName of the brewery.
     * @param breweryLocation the breweryLocation of the brewery.
     * @param breweryCountry the breweryCountry of the brewery.
     * @return the instance.
     * @throws NullPointerException if {@code breweryName} is {@code null}.
     * @throws IllegalArgumentException if {@code breweryName} is empty.
     */
    @Nonnull
    public static Brewery newBrewery(final long breweryId, @Nonnull final CharSequence breweryName,
            @Nullable final CharSequence breweryLocation, @Nullable final Country breweryCountry) {
        argumentNotEmpty(breweryName, "brewery breweryName");
        return ImmutableBrewery.newInstance(breweryId, breweryName, breweryLocation, breweryCountry);
    }

    /**
     * Returns a new immutable instance of {@link Brewery} from the specified Cursor.
     *
     * @param cursor the cursor to provide the brewery data.
     * @return the instance
     * @throws NullPointerException if {@code cursor} is {@code null}.
     * @throws IllegalArgumentException if {@code cursor} did not contain all expected columns.
     */
    @Nullable
    public static Brewery newBrewery(@Nonnull final Cursor cursor) {
        final CursorReader cr = CursorReader.of(cursor);
        final long breweryId = cr.getLong(BreweryEntry.COLUMN_ID);
        final String breweryName = cr.getString(BreweryEntry.COLUMN_NAME);
        final String breweryLocation = cr.getString(BreweryEntry.COLUMN_LOCATION);
        final Country breweryCountry = CountryFactory.newCountry(cursor);

        if (1 > breweryId || null == breweryName) {
            return null;
        }

        return newBrewery(breweryId, breweryName, breweryLocation, breweryCountry);
    }

}
