/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.beerstyle;

import android.content.ContentValues;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.data.BierverkostungContract.BeerStyleEntry;
import de.retujo.bierverkostung.data.UriToIdFunction;
import de.retujo.java.util.Maybe;
import de.retujo.java.util.ObjectUtil;

import static de.retujo.java.util.Conditions.argumentNotEmpty;

/**
 * Immutable implementation of {@link BeerStyle}.
 *
 * @since 1.0.0
 */
@Immutable
final class ImmutableBeerStyle implements BeerStyle {

    /**
     * Creator which creates instances of {@code ImmutableBeerStyle} from a Parcel.
     */
    public static final Parcelable.Creator<BeerStyle> CREATOR = new ImmutableBeerStyleCreator();

    private final long id;
    private final String name;

    private ImmutableBeerStyle(final long theId, final String theName) {
        id = theId;
        name = theName;
    }

    /**
     * Returns a new instance of {@code ImmutableBeerStyle} with the given ID and name.
     *
     * @param id the ID of the beer style.
     * @param name the name of the beer style.
     * @return the instance.
     * @throws NullPointerException if {@code name} is {@code null}.
     * @throws IllegalArgumentException if {@code name} is empty.
     */
    public static ImmutableBeerStyle newInstance(final long id, @Nonnull final CharSequence name) {
        argumentNotEmpty(name, "name");
        return new ImmutableBeerStyle(id, name.toString());
    }

    @Nonnull
    @Override
    public Maybe<Long> getId() {
        if (UriToIdFunction.NO_ID != id) {
            return Maybe.of(id);
        }
        return Maybe.empty();
    }

    @Nonnull
    @Override
    public String getName() {
        return name;
    }

    @Nonnull
    @Override
    public Uri getContentUri() {
        return BeerStyleEntry.CONTENT_URI;
    }

    @Nonnull
    @Override
    public ContentValues asContentValues() {
        final ContentValues result = new ContentValues(2);
        if (UriToIdFunction.NO_ID != id) {
            result.put(BeerStyleEntry.COLUMN_ID.toString(), id);
        }
        result.put(BeerStyleEntry.COLUMN_NAME.toString(), name);
        return result;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(final Parcel dest, final int flags) {
        // Both fields are always not null thus renouncing on use of ParcelWrapper.
        dest.writeLong(id);
        dest.writeString(name);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final ImmutableBeerStyle that = (ImmutableBeerStyle) o;

        return id == that.id && ObjectUtil.areEqual(name, that.name);
    }

    @Override
    public int hashCode() {
        return ObjectUtil.hashCode(id, name);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + " {" + "id=" + id + ", name='" + name + '\'' + "}";
    }

    /**
     * This class creates instances of {@code ImmutableBeerStyle} from a Parcel.
     *
     * @since 1.0.0
     */
    @Immutable
    private static final class ImmutableBeerStyleCreator implements Parcelable.Creator<BeerStyle> {
        @Override
        public BeerStyle createFromParcel(final Parcel source) {
            return ImmutableBeerStyle.newInstance(source.readLong(), source.readString());
        }

        @Override
        public BeerStyle[] newArray(final int size) {
            return new BeerStyle[0];
        }
    }

}
