/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.beer;

import android.app.Activity;
import android.content.ContentResolver;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.text.Editable;
import android.text.TextUtils;
import android.view.View;
import android.widget.Button;
import android.widget.MultiAutoCompleteTextView;
import android.widget.TextView;

import java.math.BigDecimal;

import javax.annotation.concurrent.NotThreadSafe;

import de.retujo.bierverkostung.R;
import de.retujo.bierverkostung.beerstyle.BeerStyle;
import de.retujo.bierverkostung.beerstyle.SelectBeerStyleActivity;
import de.retujo.bierverkostung.brewery.Brewery;
import de.retujo.bierverkostung.brewery.SelectBreweryActivity;
import de.retujo.bierverkostung.common.AbstractTextWatcher;
import de.retujo.bierverkostung.common.BaseActivity;
import de.retujo.bierverkostung.common.ButtonEnablingTextWatcher;
import de.retujo.bierverkostung.data.DbColumnArrayAdapter;
import de.retujo.bierverkostung.data.BierverkostungContract.BeerEntry;
import de.retujo.bierverkostung.data.UriToIdFunction;
import de.retujo.java.util.Maybe;
import de.retujo.java.util.Provider;

/**
 * Activity for creating a new beer or for editing an existing beer. The only mandatory property is the name; thus as
 * soon as a name is entered the button for saving becomes enabled.
 *
 * @since 1.0.0
 */
@SuppressWarnings("squid:MaximumInheritanceDepth")
@NotThreadSafe
public final class EditBeerActivity extends BaseActivity {

    /**
     * Key to retrieve the edited beer from this Activity's result Intent.
     */
    public static final String EDITED_BEER = "editedBeer";

    /**
     * Key to set an existing Beer to be edited to the Intent for starting an EditBeerActivity.
     */
    public static final String EDIT_BEER = "beerToBeEdited";

    private static final int SELECT_BREWERY_REQUEST_CODE = 160;
    private static final int SELECT_STYLE_REQUEST_CODE = 145;
    private static final String NEW_BEER_BREWERY = "newBeerBrewery";
    private static final String NEW_BEER_STYLE = "newBeerStyle";

    private TextView beerNameTextView;
    private TextView breweryTextView;
    private TextView styleTextView;
    private TextView originalWortTextView;
    private TextView alcoholTextView;
    private TextView ibuTextView;
    private MultiAutoCompleteTextView ingredientsTextView;
    private TextView specificsTextView;
    private TextView notesTextView;
    private Button saveBeerButton;
    private Runnable saveAction;

    /**
     * Constructs a new {@code EditBeerActivity} object.
     */
    public EditBeerActivity() {
        beerNameTextView = null;
        breweryTextView = null;
        styleTextView = null;
        originalWortTextView = null;
        alcoholTextView = null;
        saveBeerButton = null;
        ibuTextView = null;
        ingredientsTextView = null;
        specificsTextView = null;
        notesTextView = null;
        saveAction = null;
    }

    @Override
    public void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_edit_beer);

        initSaveBeerButton();
        initBeerNameTextView();
        initBreweryTextView();
        initStyleTextView();

        originalWortTextView = findView(R.id.edit_beer_original_wort_edit_text);
        alcoholTextView = findView(R.id.edit_beer_alcohol_edit_text);
        ibuTextView = findView(R.id.edit_beer_ibu_edit_text);
        initIngredientsTextView();
        specificsTextView = findView(R.id.edit_beer_specifics_edit_text);
        notesTextView = findView(R.id.edit_beer_notes_edit_text);

        final Intent afferentIntent = getIntent();
        final Beer beerToBeEdited = afferentIntent.getParcelableExtra(EDIT_BEER);
        if (null == beerToBeEdited) {
            setTitle(getString(R.string.add_beer_activity_name));
            restore(savedInstanceState);
            saveAction = new SaveNewBeerAction();
        } else {
            setTitle(getString(R.string.edit_beer_activity_name));
            initViewsWithBeerProperties(beerToBeEdited);
            saveAction = new SaveEditedBeerAction(beerToBeEdited);
        }
    }

    private void initSaveBeerButton() {
        saveBeerButton = findView(R.id.edit_beer_save_button);
        saveBeerButton.setEnabled(false);
    }

    private void initBeerNameTextView() {
        beerNameTextView = findView(R.id.edit_beer_name_edit_text);
        beerNameTextView.addTextChangedListener(new ButtonEnablingTextWatcher(saveBeerButton));
    }

    private void initBreweryTextView() {
        breweryTextView = findView(R.id.edit_beer_brewery_text_view);
        breweryTextView.setOnClickListener(v -> {
            final Intent selectBreweryIntent = new Intent(EditBeerActivity.this, SelectBreweryActivity.class);
            startActivityForResult(selectBreweryIntent, SELECT_BREWERY_REQUEST_CODE);
        });
    }

    private void initStyleTextView() {
        styleTextView = findView(R.id.edit_beer_style_text_view);
        styleTextView.setOnClickListener(v -> {
            final Intent selectBeerStyleIntent = new Intent(EditBeerActivity.this, SelectBeerStyleActivity.class);
            startActivityForResult(selectBeerStyleIntent, SELECT_STYLE_REQUEST_CODE);
        });
    }

    private void initIngredientsTextView() {
        ingredientsTextView = findView(R.id.edit_beer_ingredients_edit_text);
        final DbColumnArrayAdapter arrayAdapter = DbColumnArrayAdapter.getInstance(this,
                BeerEntry.CONTENT_URI_SINGLE_COLUMN, BeerEntry.COLUMN_INGREDIENTS);
        ingredientsTextView.setAdapter(arrayAdapter);
        ingredientsTextView.setTokenizer(new MultiAutoCompleteTextView.CommaTokenizer());
        ingredientsTextView.setThreshold(1);
    }

    private void restore(final Bundle savedInstanceState) {
        if (null != savedInstanceState) {
            final Brewery brewery = savedInstanceState.getParcelable(NEW_BEER_BREWERY);
            if (null != brewery) {
                breweryTextView.setText(brewery.getName());
                breweryTextView.setTag(brewery);
            }
            final BeerStyle beerStyle = savedInstanceState.getParcelable(NEW_BEER_STYLE);
            if (null != beerStyle) {
                styleTextView.setText(beerStyle.getName());
                styleTextView.setTag(beerStyle);
            }
        }
    }

    private void initViewsWithBeerProperties(final Beer beer) {
        beerNameTextView.setText(beer.getName());
        final Maybe<Brewery> breweryMaybe = beer.getBrewery();
        if (breweryMaybe.isPresent()) {
            final Brewery brewery = breweryMaybe.get();
            breweryTextView.setText(brewery.getName());
            breweryTextView.setTag(brewery);
        }
        final Maybe<BeerStyle> styleMaybe = beer.getStyle();
        if (styleMaybe.isPresent()) {
            final BeerStyle beerStyle = styleMaybe.get();
            styleTextView.setText(beerStyle.getName());
            styleTextView.setTag(beerStyle);
        }
        setTextIfPresent(alcoholTextView, beer.getAlcohol());
        setTextIfPresent(originalWortTextView, beer.getOriginalWort());
        final Maybe<Integer> ibu = beer.getIbu();
        if (ibu.isPresent()) {
           ibuTextView.setText(String.valueOf(ibu.get()));
        }
        setTextIfPresent(ingredientsTextView, beer.getIngredients());
        setTextIfPresent(specificsTextView, beer.getSpecifics());
        setTextIfPresent(notesTextView, beer.getNotes());
    }

    private static void setTextIfPresent(final TextView textView, final Maybe<String> maybe) {
        if (maybe.isPresent()) {
            textView.setText(maybe.get());
        }
    }

    @Override
    public void onPostCreate(@Nullable final Bundle savedInstanceState) {
        super.onPostCreate(savedInstanceState);
        originalWortTextView.addTextChangedListener(new CalculateAlcoholTextWatcher());
    }

    @Override
    public void onSaveInstanceState(final Bundle outState) {
        super.onSaveInstanceState(outState);

        // Save state of TextViews which are not automatically saved by Android as they are no EditTexts.
        final Brewery brewery = (Brewery) breweryTextView.getTag();
        if (null != brewery) {
            outState.putParcelable(NEW_BEER_BREWERY, brewery);
        }
        final BeerStyle style = (BeerStyle) styleTextView.getTag();
        if (null != style) {
            outState.putParcelable(NEW_BEER_STYLE, style);
        }
    }

    @Override
    protected void onActivityResult(final int requestCode, final int resultCode, final Intent data) {
        if (Activity.RESULT_OK == resultCode) {
            if (SELECT_BREWERY_REQUEST_CODE == requestCode) {
                saveSelectedBreweryAsTag(data);
            } else if (SELECT_STYLE_REQUEST_CODE == requestCode) {
                saveSelectedStyleAsTag(data);
            }
        }
    }

    private void saveSelectedBreweryAsTag(final Intent data) {
        // Save selected brewery as tag in brewery name TextView
        final Brewery selectedBrewery = data.getParcelableExtra(SelectBreweryActivity.SELECTED_BREWERY);
        breweryTextView.setText(selectedBrewery.getName());
        breweryTextView.setTag(selectedBrewery);
    }

    private void saveSelectedStyleAsTag(final Intent data) {
        // Save selected style as tag in style TextView
        final BeerStyle selectedBeerStyle = data.getParcelableExtra(SelectBeerStyleActivity.SELECTED_BEER_STYLE);
        styleTextView.setText(selectedBeerStyle.getName());
        styleTextView.setTag(selectedBeerStyle);
    }

    /**
     * Persists the beer which is derived from the entered data.
     *
     * @param view the View is ignored by this method.
     */
    public void saveBeer(final View view) {
        saveAction.run();
        finish();
    }

    private void setResult(final Beer beer) {
        final Intent resultIntent = new Intent();
        resultIntent.putExtra(EDITED_BEER, beer);
        setResult(Activity.RESULT_OK, resultIntent);
    }

    /**
     * This TextWatcher calculates the alcohol based on the original wort. This only applies if no alcohol value was
     * entered directly.
     *
     * @since 1.0.0
     */
    @NotThreadSafe
    private final class CalculateAlcoholTextWatcher extends AbstractTextWatcher {
        private boolean alcoholTextViewChangedByThis;

        private CalculateAlcoholTextWatcher() {
            alcoholTextViewChangedByThis = false;
        }

        @Override
        public void afterTextChanged(final Editable s) {
            if (!TextUtils.isEmpty(s)) {
                if (isAlcoholTextViewCurrentlyEmpty() || alcoholTextViewChangedByThis) {

                    // otherwise an alcohol value for the beer was already set by the user or loaded from database
                    calculateAlcohol(s);
                }
            } else if (alcoholTextViewChangedByThis) {
                alcoholTextView.setText(null);
            }
        }

        private boolean isAlcoholTextViewCurrentlyEmpty() {
            return TextUtils.isEmpty(alcoholTextView.getText());
        }

        private void calculateAlcohol(final CharSequence s) {
            final double originalWort = Double.parseDouble(s.toString());
            final Provider<BigDecimal> alcoholProvider = AlcoholProvider.newInstance(originalWort);
            final BigDecimal alcoholValue = alcoholProvider.get();
            if (0 < alcoholValue.compareTo(BigDecimal.ZERO)) {
                final NumberFormatter numberFormatter = NumberFormatter.newInstance(alcoholValue, 1);
                alcoholTextView.setText(numberFormatter.get());
                alcoholTextViewChangedByThis = true;
            }
        }
    }

    /**
     * This class inserts a new Beer into the database.
     *
     * @since 1.0.0
     */
    @NotThreadSafe
    private final class SaveNewBeerAction implements Runnable {
        @Override
        public void run() {
            final BeerBuilder beerBuilder = BeerBuilder.newInstance(beerNameTextView.getText())
                    .brewery((Brewery) breweryTextView.getTag())
                    .style((BeerStyle) styleTextView.getTag())
                    .originalWort(originalWortTextView.getText())
                    .alcohol(alcoholTextView.getText())
                    .ibu(ibuTextView.getText())
                    .ingredients(ingredientsTextView.getText())
                    .specifics(specificsTextView.getText())
                    .notes(notesTextView.getText());

            final Beer beerWithoutId = beerBuilder.build();

            final ContentResolver contentResolver = getContentResolver();
            final Uri uri = contentResolver.insert(beerWithoutId.getContentUri(), beerWithoutId.asContentValues());

            final Beer beerWithId = beerBuilder
                    .id(getBeerId(uri))
                    .build();

            setResult(beerWithId);
        }

        @SuppressWarnings("ConstantConditions")
        private long getBeerId(final Uri beerRowUri) {
            final UriToIdFunction uriToIdFunction = UriToIdFunction.getInstance();
            return uriToIdFunction.apply(beerRowUri).orElse(UriToIdFunction.NO_ID);
        }
    }

    /**
     * This class updates an existing Beer in the database.
     *
     * @since 1.0.0
     */
    @NotThreadSafe
    private final class SaveEditedBeerAction implements Runnable {
        private final Beer beerToBeEdited;

        private SaveEditedBeerAction(final Beer beerToBeEdited) {
            this.beerToBeEdited = beerToBeEdited;
        }

        @Override
        public void run() {
            final Long beerId = beerToBeEdited.getId().orElse(null);
            if (null != beerId) {
                final Beer editedBeer = BeerBuilder.newInstance(beerNameTextView.getText())
                        .id(beerId)
                        .brewery((Brewery) breweryTextView.getTag())
                        .style((BeerStyle) styleTextView.getTag())
                        .originalWort(originalWortTextView.getText())
                        .alcohol(alcoholTextView.getText())
                        .ibu(ibuTextView.getText())
                        .ingredients(ingredientsTextView.getText())
                        .specifics(specificsTextView.getText())
                        .notes(notesTextView.getText())
                        .build();
                final Uri uri = editedBeer.getContentUri()
                        .buildUpon()
                        .appendPath(String.valueOf(beerId))
                        .build();
                final ContentResolver contentResolver = getContentResolver();
                contentResolver.update(uri, editedBeer.asContentValues(), null, null);

                setResult(editedBeer);
            }
        }
    }

}
