/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.beer;

import android.database.Cursor;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.beerstyle.BeerStyleFactory;
import de.retujo.bierverkostung.brewery.BreweryFactory;
import de.retujo.bierverkostung.data.BierverkostungContract.BeerEntry;
import de.retujo.bierverkostung.data.Column;
import de.retujo.bierverkostung.data.CursorReader;

/**
 * A factory for immutable instances of {@link Beer}.
 *
 * @since 1.0.0
 */
@Immutable
public final class BeerFactory {

    private BeerFactory() {
        throw new AssertionError();
    }

    /**
     * Returns a new immutable instance of {@link Beer} from the specified Cursor.
     *
     * @param cursor the cursor to provide the beer data.
     * @return the instance.
     * @throws NullPointerException if {@code cursor} is {@code null}.
     * @throws IllegalArgumentException if {@code cursor} did not contain all expected columns.
     */
    @Nonnull
    public static Beer newBeer(@Nonnull final Cursor cursor) {
        final CursorReader cr = CursorReader.of(cursor);

        return BeerBuilder.newInstance(cr.getString(BeerEntry.COLUMN_NAME))
                .id(cr.getLong(BeerEntry.COLUMN_ID))
                .brewery(BreweryFactory.newBrewery(cursor))
                .style(BeerStyleFactory.newBeerStyle(cursor))
                .originalWort(getDoubleAsString(cr, BeerEntry.COLUMN_ORIGINAL_WORT))
                .alcohol(getDoubleAsString(cr, BeerEntry.COLUMN_ALCOHOL))
                .ibu(cr.getInt(BeerEntry.COLUMN_IBU))
                .ingredients(cr.getString(BeerEntry.COLUMN_INGREDIENTS))
                .specifics(cr.getString(BeerEntry.COLUMN_SPECIFICS))
                .notes(cr.getString(BeerEntry.COLUMN_NOTES))
                .build();
    }

    private static String getDoubleAsString(final CursorReader cursorReader, final Column column) {
        final double doubleValue = cursorReader.getDouble(column);
        final String doubleValueAsString = String.valueOf(doubleValue);
        if ("0.0".equals(doubleValueAsString)) {
            return null;
        }
        return doubleValueAsString;
    }

}
