/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.beer;

import android.content.ContentValues;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;

import de.retujo.bierverkostung.beerstyle.BeerStyle;
import de.retujo.bierverkostung.brewery.Brewery;
import de.retujo.bierverkostung.common.CommaDelimiterTrimmer;
import de.retujo.bierverkostung.data.BierverkostungContract.BeerEntry;
import de.retujo.bierverkostung.data.ParcelUnwrapper;
import de.retujo.bierverkostung.data.ParcelWrapper;
import de.retujo.bierverkostung.data.UriToIdFunction;
import de.retujo.java.util.Maybe;
import de.retujo.java.util.ObjectUtil;

import static android.text.TextUtils.isEmpty;
import static de.retujo.java.util.Conditions.argumentNotEmpty;
import static de.retujo.java.util.ObjectUtil.areEqual;

/**
 * A mutable builder with a fluent API for an immutable {@link Beer}.
 *
 * @since 1.0.0
 */
@NotThreadSafe
public final class BeerBuilder {

    private final String name;
    private long id;
    private Brewery brewery;
    private BeerStyle style;
    private String originalWort;
    private String alcohol;
    private int ibu;
    private String ingredients;
    private String specifics;
    private String notes;

    private BeerBuilder(final String theName) {
        name = theName;
        id = UriToIdFunction.NO_ID;
        brewery = null;
        style = null;
        originalWort = null;
        alcohol = null;
        ibu = ImmutableBeer.NULL_IBU;
        ingredients = null;
        specifics = null;
        notes = null;
    }

    /**
     * Returns a new instance of {@code BeerBuilder}.
     *
     * @param beerName the name of the beer.
     * @return the instance.
     * @throws NullPointerException if {@code beerName} is {@code null}.
     * @throws IllegalArgumentException if {@code beerName} is empty.
     */
    public static BeerBuilder newInstance(final CharSequence beerName) {
        argumentNotEmpty(beerName, "beer name");
        return new BeerBuilder(beerName.toString());
    }

    /**
     * Sets the ID of the beer.
     *
     * @param id the ID.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder id(final long id) {
        if (0 < id) {
            this.id = id;
        }
        return this;
    }

    /**
     * Sets the brewery of the beer.
     *
     * @param brewery the Brewery or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder brewery(@Nullable final Brewery brewery) {
        this.brewery = brewery;
        return this;
    }

    /**
     * Sets the style of the beer.
     *
     * @param beerStyle the style or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder style(@Nullable final BeerStyle beerStyle) {
        this.style = beerStyle;
        return this;
    }

    /**
     * Sets the original wort of the beer.
     *
     * @param originalWort the original wort or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder originalWort(@Nullable final CharSequence originalWort) {
        if (originalWort != null) {
            this.originalWort = originalWort.toString();
        }
        return this;
    }

    /**
     * Sets the original wort of the beer.
     *
     * @param originalWort the original wort.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder originalWort(final double originalWort) {
        return originalWort(String.valueOf(originalWort));
    }

    /**
     * Sets the volume alcohol of the beer.
     *
     * @param alcohol the alcohol value or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder alcohol(@Nullable final CharSequence alcohol) {
        if (alcohol != null) {
            this.alcohol = alcohol.toString();
        }
        return this;
    }

    /**
     * Sets the volume alcohol of the beer.
     *
     * @param alcohol the alcohol value.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder alcohol(final double alcohol) {
        return alcohol(String.valueOf(alcohol));
    }

    /**
     * Sets the IBU value of the beer.
     *
     * @param ibu the IBU value or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder ibu(@Nullable final CharSequence ibu) {
        if (!isEmpty(ibu)) {
            this.ibu = Integer.parseInt(ibu.toString());
        }
        return this;
    }

    /**
     * Sets the IBU value of the beer.
     *
     * @param ibu the IBU value.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder ibu(final int ibu) {
        this.ibu = ibu;
        return this;
    }

    /**
     * Sets the ingredients of the beer.
     *
     * @param ingredients the ingredients or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder ingredients(@Nullable final CharSequence ingredients) {
        if (ingredients != null) {
            this.ingredients = CommaDelimiterTrimmer.trim(ingredients);
        }
        return this;
    }

    /**
     * Sets the specifics of the beer.
     *
     * @param specifics the specifics or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder specifics(@Nullable final CharSequence specifics) {
        if (specifics != null) {
            this.specifics = specifics.toString();
        }
        return this;
    }

    /**
     * Sets the notes of the beer.
     *
     * @param notes the notes or {@code null}.
     * @return this builder instance to allow Method Chaining.
     */
    public BeerBuilder notes(@Nullable final CharSequence notes) {
        if (notes != null) {
            this.notes = notes.toString();
        }
        return this;
    }

    /**
     * Returns a new immutable instance of {@code Beer} with the properties of this builder.
     *
     * @return the new Beer.
     */
    public Beer build() {
        return new ImmutableBeer(this);
    }

    @Immutable
    static final class ImmutableBeer implements Beer {
        /**
         * Creator which creates instances of {@code ImmutableBeer} from a Parcel.
         */
        public static final Parcelable.Creator<Beer> CREATOR = new ImmutableBeerCreator();

        /**
         * This value represents a semantic {@code null} value for IBU.
         */
        private static final int NULL_IBU = 0;

        private static final int MAX_CONTENT_VALUES = 10;

        private final String name;
        private final long id;
        private final Brewery brewery;
        private final BeerStyle style;
        private final String originalWort;
        private final String alcohol;
        private final int ibu;
        private final String ingredients;
        private final String specifics;
        private final String notes;

        private ImmutableBeer(final BeerBuilder builder) {
            name = builder.name;
            id = builder.id;
            brewery = builder.brewery;
            style = builder.style;
            originalWort = builder.originalWort;
            alcohol = builder.alcohol;
            ibu = builder.ibu;
            ingredients = builder.ingredients;
            specifics = builder.specifics;
            notes = builder.notes;
        }

        @Nonnull
        @Override
        public String getName() {
            return name;
        }

        @Nonnull
        @Override
        public Maybe<Long> getId() {
            if (UriToIdFunction.NO_ID == id) {
                return Maybe.empty();
            }
            return Maybe.of(id);
        }

        @Nonnull
        @Override
        public Maybe<Brewery> getBrewery() {
            return Maybe.ofNullable(brewery);
        }

        @Nonnull
        @Override
        public Maybe<BeerStyle> getStyle() {
            return Maybe.ofNullable(style);
        }

        @Nonnull
        @Override
        public Maybe<String> getOriginalWort() {
            return Maybe.ofNullable(originalWort);
        }

        @Nonnull
        @Override
        public Maybe<String> getAlcohol() {
            return Maybe.ofNullable(alcohol);
        }

        @Nonnull
        @Override
        public Maybe<Integer> getIbu() {
            if (NULL_IBU != ibu) {
                return Maybe.of(ibu);
            }
            return Maybe.empty();
        }

        @Nonnull
        @Override
        public Maybe<String> getIngredients() {
            return Maybe.ofNullable(ingredients);
        }

        @Nonnull
        @Override
        public Maybe<String> getSpecifics() {
            return Maybe.ofNullable(specifics);
        }

        @Nonnull
        @Override
        public Maybe<String> getNotes() {
            return Maybe.ofNullable(notes);
        }

        @Nonnull
        @Override
        public Uri getContentUri() {
            return BeerEntry.CONTENT_URI;
        }

        @Nonnull
        @Override
        public ContentValues asContentValues() {
            final ContentValues result = new ContentValues(MAX_CONTENT_VALUES);
            result.put(BeerEntry.COLUMN_NAME.toString(), name);
            if (UriToIdFunction.NO_ID != id) {
                result.put(BeerEntry.COLUMN_ID.toString(), id);
            }
            if (null != brewery) {
                result.put(BeerEntry.COLUMN_BREWERY_ID.toString(), brewery.getId().orElse(UriToIdFunction.NO_ID));
            }
            if (null != style) {
                result.put(BeerEntry.COLUMN_STYLE_ID.toString(), style.getId().orElse(UriToIdFunction.NO_ID));
            }
            if (!isEmpty(originalWort)) {
                result.put(BeerEntry.COLUMN_ORIGINAL_WORT.toString(), Double.parseDouble(originalWort));
            }
            if (!isEmpty(alcohol)) {
                result.put(BeerEntry.COLUMN_ALCOHOL.toString(), Double.parseDouble(alcohol));
            }
            result.put(BeerEntry.COLUMN_IBU.toString(), ibu);
            if (!isEmpty(ingredients)) {
                result.put(BeerEntry.COLUMN_INGREDIENTS.toString(), ingredients);
            }
            if (!isEmpty(specifics)) {
                result.put(BeerEntry.COLUMN_SPECIFICS.toString(), specifics);
            }
            if (!isEmpty(notes)) {
                result.put(BeerEntry.COLUMN_NOTES.toString(), notes);
            }
            return result;
        }

        @Override
        public int describeContents() {
            return 0;
        }

        @Override
        public void writeToParcel(final Parcel dest, final int flags) {
            ParcelWrapper.of(dest)
                    .wrap(id)
                    .wrap(name)
                    .wrap(brewery)
                    .wrap(style)
                    .wrap(originalWort)
                    .wrap(alcohol)
                    .wrap(ibu)
                    .wrap(ingredients)
                    .wrap(specifics)
                    .wrap(notes);
        }

        @SuppressWarnings("squid:S1067")
        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            final ImmutableBeer that = (ImmutableBeer) o;

            return id == that.id
                    && areEqual(name, that.name)
                    && areEqual(brewery, that.brewery)
                    && areEqual(style, that.style)
                    && areEqual(originalWort, that.originalWort)
                    && areEqual(alcohol, that.alcohol)
                    && ibu == that.ibu
                    && areEqual(ingredients, that.ingredients)
                    && areEqual(specifics, that.specifics)
                    && areEqual(notes, that.notes);

        }

        @Override
        public int hashCode() {
            return ObjectUtil.hashCode(name, id, brewery, style, originalWort, alcohol, ibu, ingredients, specifics,
                    notes);
        }

        @Override
        public String toString() {
            return getClass().getSimpleName() + " {"
                    + "name='" + name + '\''
                    + ", id=" + id
                    + ", brewery=" + brewery
                    + ", style=" + style
                    + ", originalWort='" + originalWort + '\''
                    + ", alcohol='" + alcohol + '\''
                    + ", ibu=" + ibu
                    + ", ingredients='" + ingredients + '\''
                    + ", specifics='" + specifics + '\''
                    + ", notes='" + notes + '\''
                    + "}";
        }
    }

    /**
     * This class creates instances of {@code ImmutableBeer} from a Parcel.
     *
     * @since 1.0.0
     */
    @Immutable
    private static final class ImmutableBeerCreator implements Parcelable.Creator<Beer> {
        @Override
        public Beer createFromParcel(final Parcel source) {
            final ParcelUnwrapper unwrapper = ParcelUnwrapper.of(source);
            final long readId = unwrapper.unwrapLong();
            final String readName = unwrapper.unwrapString();

            return BeerBuilder.newInstance(readName)
                    .id(readId)
                    .brewery(unwrapper.unwrapParcelable())
                    .style(unwrapper.unwrapParcelable())
                    .originalWort(unwrapper.unwrapString())
                    .alcohol(unwrapper.unwrapString())
                    .ibu(unwrapper.unwrapInt())
                    .ingredients(unwrapper.unwrapString())
                    .specifics(unwrapper.unwrapString())
                    .notes(unwrapper.unwrapString())
                    .build();
        }

        @Override
        public Beer[] newArray(final int size) {
            return new Beer[size];
        }
    }

}
