/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import android.content.ComponentName;
import android.content.ContentResolver;
import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.pm.ProviderInfo;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.net.Uri;
import android.support.test.InstrumentationRegistry;
import android.support.test.runner.AndroidJUnit4;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;

import de.retujo.bierverkostung.data.BierverkostungContract.BeerEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.BeerStyleEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.BreweryEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.CountryEntry;
import de.retujo.bierverkostung.data.BierverkostungContract.TastingEntry;

import static junit.framework.Assert.assertEquals;
import static junit.framework.Assert.assertNotNull;
import static junit.framework.Assert.assertTrue;
import static junit.framework.Assert.fail;

/**
 * Unit test for {@link BierverkostungContentProvider}.
 *
 * @since 1.0.0
 */
@RunWith(AndroidJUnit4.class)
@Ignore
public final class BierverkostungContentProviderTest {

    private Context appContext = null;
    private SQLiteOpenHelper dbHelper = null;
    private ContentResolver contentResolver = null;

    /** */
    @Before
    public void setUp() {
        appContext = InstrumentationRegistry.getTargetContext();
        contentResolver = appContext.getContentResolver();
        dbHelper = new BierverkostungDbHelper(appContext);

        final SQLiteDatabase database = dbHelper.getWritableDatabase();
        database.delete(TastingEntry.TABLE.toString(), null, null);
        database.delete(BeerEntry.TABLE.toString(), null, null);
        database.delete(BreweryEntry.TABLE.toString(), null, null);
        database.delete(CountryEntry.TABLE.toString(), null, null);
        database.delete(BeerStyleEntry.TABLE.toString(), null, null);
    }

    /**
     * This test checks to make sure that the content provider is registered correctly in the
     * AndroidManifest file. If it fails, the AndroidManifest should be checked to see if the
     * <provider/> tag is dded and that the android:authorities attribute is properly specified.
     */
    @Test
    public void testProviderRegistry() {

        /*
         * A ComponentName is an identifier for a specific application component, such as an
         * Activity, ContentProvider, BroadcastReceiver, or a Service.
         *
         * Two pieces of information are required to identify a component: the package (a String)
         * it exists in, and the class (a String) name inside of that package.
         *
         * We will use the ComponentName for our ContentProvider class to ask the system
         * information about the ContentProvider, specifically, the authority under which it is
         * registered.
         */
        final String packageName = appContext.getPackageName();
        final String contentProviderClassName = BierverkostungContentProvider.class.getName();
        final ComponentName componentName = new ComponentName(packageName, contentProviderClassName);

        try {

            /*
             * Get a reference to the package manager. The package manager allows us to access
             * information about packages installed on a particular device. In this case, we're
             * going to use it to get some information about our ContentProvider under test.
             */
            final PackageManager packageManager = appContext.getPackageManager();

            /* The ProviderInfo will contain the authority, which is what we want to test */
            final ProviderInfo providerInfo = packageManager.getProviderInfo(componentName, 0);
            final String actualAuthority = providerInfo.authority;
            final String expectedAuthority = packageName;

            /* Make sure that the registered authority matches the authority from the Contract */
            final String incorrectAuthority = "BierverkostungContentProvider registered with authority: "
                    + actualAuthority + " instead of expected authority: " + expectedAuthority;
            assertEquals(incorrectAuthority, actualAuthority, expectedAuthority);
        } catch (final PackageManager.NameNotFoundException e) {
            final String providerNotRegisteredAtAll = "BierverkostungContentProvider not registered at "
                    + appContext.getPackageName();
            /*
             * This exception is thrown if the ContentProvider hasn't been registered with the
             * manifest at all. If this is the case, you need to double check your
             * AndroidManifest file
             */
            fail(providerNotRegisteredAtAll);
        }
    }

    /** */
    @Test
    public void testInsertBeerStyle() {
        final ContentValues beerStyleValues = new ContentValues();
        beerStyleValues.put(BeerStyleEntry.COLUMN_NAME.toString(), "Hefe-Weißbier");

        final Uri uri = contentResolver.insert(BeerStyleEntry.CONTENT_URI, beerStyleValues);
        final Uri expectedUri = ContentUris.withAppendedId(BeerStyleEntry.CONTENT_URI, 1);

        assertEquals("Unable to insert item through Provider", expectedUri, uri);
    }

    /**
     * Tests inserting a single row of data via a ContentResolver into Countries table.
     */
    @Test
    public void testInsertCountry() {
        final ContentValues countryValues = new ContentValues();
        countryValues.put(CountryEntry.COLUMN_NAME.toString(), "Deutschland");

        final Uri uri = contentResolver.insert(CountryEntry.CONTENT_URI, countryValues);
        final Uri expectedUri = ContentUris.withAppendedId(CountryEntry.CONTENT_URI, 1);

        assertEquals("Unable to insert item through Provider", expectedUri, uri);
    }

    @Test
    public void testQueryBrewery() {
        final String breweryName = "Hausbraxatur Fickel";
        final String breweryLocation = "Forheim";
        final int countryId = 1;

        final SQLiteDatabase database = dbHelper.getWritableDatabase();

        final ContentValues contentValues = new ContentValues();
        contentValues.put(BreweryEntry.COLUMN_NAME.toString(), breweryName);
        contentValues.put(BreweryEntry.COLUMN_LOCATION.toString(), breweryLocation);
        contentValues.put(BreweryEntry.COLUMN_COUNTRY_ID.toString(), countryId);

        final long breweryRowId = database.insert(BreweryEntry.TABLE.toString(), null, contentValues);

        assertTrue("Unable to insert directly into the database", breweryRowId != -1);

        database.close();

        final ContentResolver contentResolver = appContext.getContentResolver();
        final Cursor breweryCursor = contentResolver.query(BreweryEntry.CONTENT_URI, null, null, null, null);

        assertNotNull("Query failed to return a valid Cursor", breweryCursor);

        breweryCursor.close();
    }

}
