/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *  strict-local
 * @format
 */
"use strict";

function _slicedToArray(arr, i) {
  return (
    _arrayWithHoles(arr) || _iterableToArrayLimit(arr, i) || _nonIterableRest()
  );
}

function _nonIterableRest() {
  throw new TypeError("Invalid attempt to destructure non-iterable instance");
}

function _iterableToArrayLimit(arr, i) {
  var _arr = [];
  var _n = true;
  var _d = false;
  var _e = undefined;
  try {
    for (
      var _i = arr[Symbol.iterator](), _s;
      !(_n = (_s = _i.next()).done);
      _n = true
    ) {
      _arr.push(_s.value);
      if (i && _arr.length === i) break;
    }
  } catch (err) {
    _d = true;
    _e = err;
  } finally {
    try {
      if (!_n && _i["return"] != null) _i["return"]();
    } finally {
      if (_d) throw _e;
    }
  }
  return _arr;
}

function _arrayWithHoles(arr) {
  if (Array.isArray(arr)) return arr;
}

const PRAGMA = "//# offsetTable=";

function sliceModules(offsetTable, str, startOffset) {
  const modules = [];
  let offset = startOffset;

  for (const _ref of offsetTable) {
    var _ref2 = _slicedToArray(_ref, 2);

    const id = _ref2[0];
    const length = _ref2[1];
    modules.push([id, str.slice(offset, offset + length)]); // Modules are separated by a line break.

    offset += length + 1;
  }

  return [offset, modules];
}
/**
 * Parses a bundle from an embedded delta bundle.
 */

function stringToBundle(str) {
  // TODO(T34761233): This is a potential security risk!
  // It is prone to failure or exploit if the pragma isn't present at
  // the end of the bundle, since it will also match any string that
  // contains it.
  //
  // The only way to be sure that the pragma is a comment is to
  // implement a simple tokenizer, and making sure that our pragma is:
  // * at the beginning of a line (whitespace notwithstanding)
  // * not inside of a multiline comment (/* */);
  // * not inside of a multiline string (`` or escaped "").
  //
  // One way to avoid this would be to
  // require the comment to be either at the very start or at the very
  // end of the bundle.
  const pragmaIndex = str.lastIndexOf(PRAGMA);

  if (pragmaIndex === -1) {
    throw new Error("stringToBundle: Pragma not found in string bundle.");
  }

  const tableStart = pragmaIndex + PRAGMA.length;
  const tableEnd = str.indexOf("\n", tableStart);
  const offsetTable = JSON.parse(
    str.slice(tableStart, tableEnd === -1 ? str.length : tableEnd)
  );
  const pre = str.slice(0, offsetTable.pre);

  const _sliceModules = sliceModules(
      offsetTable.modules,
      str, // There's a line break after the pre segment.
      offsetTable.pre + 1
    ),
    _sliceModules2 = _slicedToArray(_sliceModules, 2),
    offset = _sliceModules2[0],
    modules = _sliceModules2[1]; // We technically don't need the bundle post segment length, since it should
  // normally stop right before the pragma.

  const post = str.slice(offset, offset + offsetTable.post);
  const bundle = {
    base: true,
    revisionId: offsetTable.revisionId,
    pre,
    post,
    modules
  };
  return bundle;
}

module.exports = stringToBundle;
