import { isValidElementType } from 'react-is';

import invariant from '../utils/invariant';

/**
 * Make sure the config passed e.g. to StackRouter, TabRouter has
 * the correct format, and throw a clear error if it doesn't.
 */
function validateRouteConfigMap(routeConfigs) {
  const routeNames = Object.keys(routeConfigs);
  invariant(
    routeNames.length > 0,
    'Please specify at least one route when configuring a navigator.'
  );

  routeNames.forEach(routeName => {
    const routeConfig = routeConfigs[routeName];
    const screenComponent = getScreenComponent(routeConfig);

    if (
      !screenComponent ||
      (!isValidElementType(screenComponent) && !routeConfig.getScreen)
    ) {
      throw new Error(`The component for route '${routeName}' must be a React component. For example:

import MyScreen from './MyScreen';
...
${routeName}: MyScreen,
}

You can also use a navigator:

import MyNavigator from './MyNavigator';
...
${routeName}: MyNavigator,
}`);
    }

    if (routeConfig.screen && routeConfig.getScreen) {
      throw new Error(
        `Route '${routeName}' should declare a screen or a getScreen, not both.`
      );
    }
  });
}

function getScreenComponent(routeConfig) {
  if (!routeConfig) {
    return null;
  }

  return routeConfig.screen ? routeConfig.screen : routeConfig;
}

export default validateRouteConfigMap;
