'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

exports.default = getChildEventSubscriber;
/*
 * This is used to extract one children's worth of events from a stream of navigation action events
 *
 * Based on the 'action' events that get fired for this navigation state, this utility will fire
 * focus and blur events for this child
 */
function getChildEventSubscriber(addListener, key, initialLastFocusEvent = 'didBlur') {
  var actionSubscribers = new Set();
  var willFocusSubscribers = new Set();
  var didFocusSubscribers = new Set();
  var willBlurSubscribers = new Set();
  var didBlurSubscribers = new Set();
  var refocusSubscribers = new Set();

  var removeAll = () => {
    [actionSubscribers, willFocusSubscribers, didFocusSubscribers, willBlurSubscribers, didBlurSubscribers, refocusSubscribers].forEach(set => set.clear());

    upstreamSubscribers.forEach(subs => subs && subs.remove());
  };

  var getChildSubscribers = evtName => {
    switch (evtName) {
      case 'action':
        return actionSubscribers;
      case 'willFocus':
        return willFocusSubscribers;
      case 'didFocus':
        return didFocusSubscribers;
      case 'willBlur':
        return willBlurSubscribers;
      case 'didBlur':
        return didBlurSubscribers;
      case 'refocus':
        return refocusSubscribers;
      default:
        return null;
    }
  };

  var emit = (type, payload) => {
    var payloadWithType = _extends({}, payload, { type });
    var subscribers = getChildSubscribers(type);
    subscribers && subscribers.forEach(subs => {
      subs(payloadWithType);
    });
  };

  // lastFocusEvent keeps track of focus state for one route. First we assume
  // we are blurred. If we are focused on initialization, the first 'action'
  // event will cause onFocus+willFocus events because we had previously been
  // considered blurred
  var lastFocusEvent = initialLastFocusEvent;

  var upstreamEvents = ['willFocus', 'didFocus', 'willBlur', 'didBlur', 'refocus', 'action'];

  var upstreamSubscribers = upstreamEvents.map(eventName => addListener(eventName, payload => {
    if (eventName === 'refocus') {
      emit(eventName, payload);
      return;
    }

    var { state, lastState, action } = payload;
    var lastRoutes = lastState && lastState.routes;
    var routes = state && state.routes;

    // const lastFocusKey =
    //   lastState && lastState.routes && lastState.routes[lastState.index].key;
    var focusKey = routes && routes[state.index].key;

    var isChildFocused = focusKey === key;
    var lastRoute = lastRoutes && lastRoutes.find(route => route.key === key);
    var newRoute = routes && routes.find(route => route.key === key);
    var childPayload = {
      context: `${key}:${action.type}_${payload.context || 'Root'}`,
      state: newRoute,
      lastState: lastRoute,
      action,
      type: eventName
    };
    var isTransitioning = !!state && state.isTransitioning;

    var previouslylastFocusEvent = lastFocusEvent;

    if (lastFocusEvent === 'didBlur') {
      // The child is currently blurred. Look for willFocus conditions
      if (eventName === 'willFocus' && isChildFocused) {
        emit(lastFocusEvent = 'willFocus', childPayload);
      } else if (eventName === 'action' && isChildFocused) {
        emit(lastFocusEvent = 'willFocus', childPayload);
      }
    }
    if (lastFocusEvent === 'willFocus') {
      // We are currently mid-focus. Look for didFocus conditions.
      // If state.isTransitioning is false, this child event happens immediately after willFocus
      if (eventName === 'didFocus' && isChildFocused && !isTransitioning) {
        emit(lastFocusEvent = 'didFocus', childPayload);
      } else if (eventName === 'action' && isChildFocused && !isTransitioning) {
        emit(lastFocusEvent = 'didFocus', childPayload);
      }
    }

    if (lastFocusEvent === 'didFocus') {
      // The child is currently focused. Look for blurring events
      if (!isChildFocused) {
        // The child is no longer focused within this navigation state
        emit(lastFocusEvent = 'willBlur', childPayload);
      } else if (eventName === 'willBlur') {
        // The parent is getting a willBlur event
        emit(lastFocusEvent = 'willBlur', childPayload);
      } else if (eventName === 'action' && previouslylastFocusEvent === 'didFocus') {
        // While focused, pass action events to children for grandchildren focus
        emit('action', childPayload);
      }
    }

    if (lastFocusEvent === 'willBlur') {
      // The child is mid-blur. Wait for transition to end
      if (eventName === 'action' && !isChildFocused && !isTransitioning) {
        // The child is done blurring because transitioning is over, or isTransitioning
        // never began and didBlur fires immediately after willBlur
        emit(lastFocusEvent = 'didBlur', childPayload);
      } else if (eventName === 'didBlur') {
        // Pass through the parent didBlur event if it happens
        emit(lastFocusEvent = 'didBlur', childPayload);
      } else if (eventName === 'action' && isChildFocused && !isTransitioning) {
        emit(lastFocusEvent = 'didFocus', childPayload);
      } else if (eventName === 'action' && isChildFocused && isTransitioning) {
        emit(lastFocusEvent = 'willFocus', childPayload);
      }
    }

    if (lastFocusEvent === 'didBlur' && !newRoute) {
      removeAll();
    }
  }));

  return {
    addListener(eventName, eventHandler) {
      var subscribers = getChildSubscribers(eventName);
      if (!subscribers) {
        throw new Error(`Invalid event name "${eventName}"`);
      }
      subscribers.add(eventHandler);
      var remove = () => {
        subscribers.delete(eventHandler);
      };
      return { remove };
    },
    emit(eventName, payload) {
      if (eventName !== 'refocus') {
        console.error(`navigation.emit only supports the 'refocus' event currently.`);
        return;
      }
      emit(eventName, payload);
    }
  };
}