'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPathParser = exports.urlToPathAndParams = exports.getParamsFromPath = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; /* eslint-disable import/no-commonjs */

var _pathToRegexp = require('path-to-regexp');

var _pathToRegexp2 = _interopRequireDefault(_pathToRegexp);

var _NavigationActions = require('../NavigationActions');

var _NavigationActions2 = _interopRequireDefault(_NavigationActions);

var _invariant = require('../utils/invariant');

var _invariant2 = _interopRequireDefault(_invariant);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var queryString = require('query-string');

var getParamsFromPath = exports.getParamsFromPath = (inputParams, pathMatch, pathMatchKeys) => {
  var params = pathMatch.slice(1).reduce(
  // iterate over matched path params
  (paramsOut, matchResult, i) => {
    var key = pathMatchKeys[i];
    if (!key || key.asterisk) {
      return paramsOut;
    }
    var paramName = key.name;

    var decodedMatchResult = void 0;
    if (matchResult) {
      try {
        decodedMatchResult = decodeURIComponent(matchResult);
      } catch (e) {
        // ignore `URIError: malformed URI`
      }
    }

    paramsOut[paramName] = decodedMatchResult || matchResult;
    return paramsOut;
  }, _extends({}, inputParams));
  return params;
};
var getRestOfPath = (pathMatch, pathMatchKeys) => {
  var rest = pathMatch[pathMatchKeys.findIndex(k => k.asterisk) + 1];
  return rest;
};

var urlToPathAndParams = exports.urlToPathAndParams = (url, uriPrefix) => {
  var searchMatch = url.match(/^(.*)\?(.*)$/);
  var params = searchMatch ? queryString.parse(searchMatch[2]) : {};
  var urlWithoutSearch = searchMatch ? searchMatch[1] : url;
  var delimiter = uriPrefix || '://';
  var path = urlWithoutSearch.split(delimiter)[1];
  if (path === undefined) {
    path = urlWithoutSearch;
  }
  if (path === '/') {
    path = '';
  }
  if (path[path.length - 1] === '/') {
    path = path.slice(0, -1);
  }
  return {
    path,
    params
  };
};

var createPathParser = exports.createPathParser = (childRouters, routeConfigs, { paths: pathConfigs = {}, disableRouteNamePaths }) => {
  var pathsByRouteNames = {};
  var paths = [];

  // Build pathsByRouteNames, which includes a regex to match paths for each route. Keep in mind, the regex will pass for the route and all child routes. The code that uses pathsByRouteNames will need to also verify that the child router produces an action, in the case of isPathMatchable false (a null path).
  Object.keys(childRouters).forEach(routeName => {
    var pathPattern = void 0;

    // First check for paths on the router, then check the route config
    if (pathConfigs[routeName] !== undefined) {
      pathPattern = pathConfigs[routeName];
    } else {
      pathPattern = routeConfigs[routeName].path;
    }

    if (pathPattern === undefined) {
      // If the user hasn't specified a path at all nor disableRouteNamePaths, then we assume the routeName is an appropriate path
      pathPattern = disableRouteNamePaths ? null : routeName;
    }

    (0, _invariant2.default)(pathPattern === null || typeof pathPattern === 'string', `Route path for ${routeName} must be specified as a string, or null.`);

    // the path may be specified as null, which is similar to empty string because it allows child routers to handle the action, but it will not match empty paths
    var isPathMatchable = pathPattern !== null;
    // pathPattern is a string with inline params, such as people/:id/*foo
    var exactReKeys = [];
    var exactRe = isPathMatchable ? (0, _pathToRegexp2.default)(pathPattern, exactReKeys) : null;
    var extendedPathReKeys = [];
    var isWildcard = pathPattern === '' || !isPathMatchable;
    var extendedPathRe = (0, _pathToRegexp2.default)(isWildcard ? '*' : `${pathPattern}/*`, extendedPathReKeys);

    pathsByRouteNames[routeName] = {
      exactRe,
      exactReKeys,
      extendedPathRe,
      extendedPathReKeys,
      isWildcard,
      toPath: pathPattern === null ? () => '' : (0, _pathToRegexp.compile)(pathPattern)
    };
  });

  paths = Object.entries(pathsByRouteNames);

  var getActionForPathAndParams = (pathToResolve = '', inputParams = {}) => {
    // Attempt to match `pathToResolve` with a route in this router's routeConfigs, deferring to child routers

    for (var [routeName, path] of paths) {
      var { exactRe, exactReKeys, extendedPathRe, extendedPathReKeys } = path;
      var childRouter = childRouters[routeName];

      var exactMatch = exactRe && exactRe.exec(pathToResolve);

      if (exactMatch && exactMatch.length) {
        var extendedMatch = extendedPathRe && extendedPathRe.exec(pathToResolve);
        var childAction = null;
        if (extendedMatch && childRouter) {
          var restOfPath = getRestOfPath(extendedMatch, extendedPathReKeys);
          childAction = childRouter.getActionForPathAndParams(restOfPath, inputParams);
        }

        return _NavigationActions2.default.navigate({
          routeName,
          params: getParamsFromPath(inputParams, exactMatch, exactReKeys),
          action: childAction
        });
      }
    }

    for (var [_routeName, _path] of paths) {
      var { extendedPathRe: _extendedPathRe, extendedPathReKeys: _extendedPathReKeys } = _path;
      var _childRouter = childRouters[_routeName];

      var _extendedMatch = _extendedPathRe && _extendedPathRe.exec(pathToResolve);

      if (_extendedMatch && _extendedMatch.length) {
        var _restOfPath = getRestOfPath(_extendedMatch, _extendedPathReKeys);
        var _childAction = null;
        if (_childRouter) {
          _childAction = _childRouter.getActionForPathAndParams(_restOfPath, inputParams);
        }
        if (!_childAction) {
          continue;
        }
        return _NavigationActions2.default.navigate({
          routeName: _routeName,
          params: getParamsFromPath(inputParams, _extendedMatch, _extendedPathReKeys),
          action: _childAction
        });
      }
    }

    return null;
  };
  var getPathAndParamsForRoute = route => {
    var { routeName, params } = route;
    var childRouter = childRouters[routeName];
    var { toPath, exactReKeys } = pathsByRouteNames[routeName];
    var subPath = toPath(params);
    var nonPathParams = {};
    if (params) {
      Object.keys(params).filter(paramName => !exactReKeys.find(k => k.name === paramName)).forEach(paramName => {
        nonPathParams[paramName] = params[paramName];
      });
    }
    if (childRouter) {
      // If it has a router it's a navigator.
      // If it doesn't have router it's an ordinary React component.
      var child = childRouter.getPathAndParamsForState(route);
      return {
        path: subPath ? `${subPath}/${child.path}` : child.path,
        params: child.params ? _extends({}, nonPathParams, child.params) : nonPathParams
      };
    }
    return {
      path: subPath,
      params: nonPathParams
    };
  };
  return { getActionForPathAndParams, getPathAndParamsForRoute };
};