/*
 * Copyright 2011 Licel LLC.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.licel.jcardsim.crypto;

import javacard.security.HMACKey;
import javacard.security.Key;
import javacard.security.KeyBuilder;
import javacard.security.Signature;
import junit.framework.TestCase;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.encoders.Hex;

/**
 * Test for
 * <code>SymmetricSignatureImpl</code>
 * Test data from NXP JCOP31-36/41 JavaCard
 */
public class SymmetricSignatureImplTest extends TestCase {

    // padded etalon message
    String MESSAGE_8 = "C899464893435BC8";
    // etalon message without padding
    String MESSAGE_15 = "C46A3D01F5494013F9DFF3C5392C64";
    // etalon des key
    String DES_KEY = "71705866C930F0AE";
    // etalon 3des key
    String DES3_KEY = "B1891A49B2EA69F21245D4A51DD132E24F247FAC6D97F007";
    // etalon IV vector
    String IV = "F8D7DB2B902855A3";
    // MESSAGE_15 MAC by card (DES key)
    String[] DES_MAC_15 = new String[]{
        // ALG_DES_MAC8_ISO9797_M1
        "2CC3B23F98A404FE",
        // ALG_DES_MAC8_ISO9797_M2
        "D7863C7D58BD53F4"
    };
    // MESSAGE_15 MAC by card (3DES key)
    String[] DES3_MAC_15 = new String[]{
        // ALG_DES_MAC8_ISO9797_M1
        "165DD117D24198B1",
        // ALG_DES_MAC8_ISO9797_M2
        "C8D247D6209E2E44",
        // ALG_DES_MAC8_ISO9797_1_M2_ALG3
        "706CAC8246DE7427",
        };
    // MESSAGE_15 MAC by card (DES key) with non-zero IV
    String[] DES_MAC_15_IV = new String[]{
        // ALG_DES_MAC8_ISO9797_M1
        "86D64C88EFE70383",
        // ALG_DES_MAC8_ISO9797_M1
        "1B66F319FA735F75",};
    // MESSAGE_15 MAC by card (3DES key) with non-zero IV
    String[] DES3_MAC_15_IV = new String[]{
        // ALG_DES_MAC8_ISO9797_M1
        "CC4EBC98BFFFAC21",
        // ALG_DES_MAC8_ISO9797_M2
        "EC17707C14FA1344",};
    // MESSAGE_8 MAC by card (DES key)
    String[] DES_MAC_8 = new String[]{
        // ALG_DES_MAC8_NOPAD
        "8E5ABFB9D5F015EE",};
    // MESSAGE_8 MAC by card (3DES key)
    String[] DES3_MAC_8 = new String[]{
        // ALG_DES_MAC8_NOPAD
        "DB3543BCBB4EAD86",};
    // MESSAGE_8 MAC by card (DES key)  with non-zero IV
    String[] DES_MAC_8_IV = new String[]{
        // ALG_DES_MAC8_NOPAD
        "3CE9E2657AFCE8B6"
    };
    // MESSAGE_8 MAC by card (3DES key)  with non-zero IV
    String[] DES3_MAC_8_IV = new String[]{
        // ALG_DES_MAC8_NOPAD
        "81B2369E2773858F"
    };
    // etalon message for AES-CBC
    String MESSAGE_16 = "6BC1BEE22E409F96E93D7E117393172A";
    // etalon aes 128 key
    String AES_128_KEY = "2B7E151628AED2A6ABF7158809CF4F3C";
    // etalon mac
    String AES_CBC_MAC = "3AD77BB40D7A3660A89ECAF32466EF97";
    // hmac test vectors
    // {SIGNATUTE_ID, ..}
    byte[] HMAC_IDS = {
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_RIPEMD160, 
                           Signature.ALG_HMAC_MD5, 
                           Signature.ALG_HMAC_MD5,
                           Signature.ALG_HMAC_MD5,
                           Signature.ALG_HMAC_MD5,
                           Signature.ALG_HMAC_MD5,
                           Signature.ALG_HMAC_MD5,
                           Signature.ALG_HMAC_MD5,        
                           Signature.ALG_HMAC_SHA1, 
                           Signature.ALG_HMAC_SHA1,
                           Signature.ALG_HMAC_SHA1,
                           Signature.ALG_HMAC_SHA1,
                           Signature.ALG_HMAC_SHA1,
                           Signature.ALG_HMAC_SHA1,
                           Signature.ALG_HMAC_SHA1,
                           Signature.ALG_HMAC_SHA_256,
                           Signature.ALG_HMAC_SHA_384,
                           Signature.ALG_HMAC_SHA_512,
                           Signature.ALG_HMAC_SHA_256,
                           Signature.ALG_HMAC_SHA_384,
                           Signature.ALG_HMAC_SHA_512,
                           Signature.ALG_HMAC_SHA_256,
                           Signature.ALG_HMAC_SHA_384,
                           Signature.ALG_HMAC_SHA_512,
                           Signature.ALG_HMAC_SHA_256,
                           Signature.ALG_HMAC_SHA_384,
                           Signature.ALG_HMAC_SHA_512,
                           Signature.ALG_HMAC_SHA_256,
                           Signature.ALG_HMAC_SHA_384,
                           Signature.ALG_HMAC_SHA_512,
                           Signature.ALG_HMAC_SHA_256,
                           Signature.ALG_HMAC_SHA_384,
                           Signature.ALG_HMAC_SHA_512,
    };
    // {KEY, TEXT, MAC}, ..
    String[][] HMAC_TESTS = {
        // RFC - 2286 HMAC-RIPEMD160 test case 1
        {"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B",
         "4869205468657265",
         "24CB4BD67D20FC1A5D2ED7732DCC39377F0A5668"},
        // RFC - 2286 HMAC-RIPEMD160 test case 2
        {"4A656665",
         "7768617420646F2079612077616E7420666F72206E6F7468696E673F",
         "DDA6C0213A485A9E24F4742064A7F033B43C4069"},
        // RFC - 2286 HMAC-RIPEMD160 test case 3
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD",
         "B0B105360DE759960AB4F35298E116E295D8E7C1"},
        // RFC - 2286 HMAC-RIPEMD160 test case 4
        {"0102030405060708090A0B0C0D0E0F10111213141516171819",
         "CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD",
         "D5CA862F4D21D5E610E18B4CF1BEB97A4365ECF4"},
        // RFC - 2286 HMAC-RIPEMD160 test case 5
        {"0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C",
         "546573742057697468205472756E636174696F6E",
         "7619693978F91D90539AE786500FF3D8E0518E39"},
        // RFC - 2286 HMAC-RIPEMD160 test case 6
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B6579202D2048617368204B6579204669727374",
         "6466CA07AC5EAC29E1BD523E5ADA7605B791FD8B"},
        // RFC - 2286 HMAC-RIPEMD160 test case 7
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B657920616E64204C6172676572205468616E204F6E6520426C6F636B2D53697A652044617461",
         "69EA60798D71616CCE5FD0871E23754CD75D5A0A"},
        // RFC - 2202 HMAC-MD5 test case 1
        {"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B",
         "4869205468657265",
         "9294727A3638BB1C13F48EF8158BFC9D"},
        // RFC - 2202 HMAC-MD5 test case 2
        {"4A656665",
         "7768617420646F2079612077616E7420666F72206E6F7468696E673F",
         "750C783E6AB0B503EAA86E310A5DB738"},
        // RFC - 2202 HMAC-MD5 test case 3
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD",
         "56BE34521D144C88DBB8C733F0E8B3F6"},
        // RFC - 2202 HMAC-MD5 test case 4
        {"0102030405060708090A0B0C0D0E0F10111213141516171819",
         "CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD",
         "697EAF0ACA3A3AEA3A75164746FFAA79"},
        // RFC - 2202 HMAC-MD5 test case 5
        {"0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C",
         "546573742057697468205472756E636174696F6E",
         "56461EF2342EDC00F9BAB995690EFD4C"},
        // RFC - 2202 HMAC-MD5 test case 6
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B6579202D2048617368204B6579204669727374",
         "6B1AB7FE4BD7BF8F0B62E6CE61B9D0CD"},
        // RFC - 2202 HMAC-MD5 test case 7
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B657920616E64204C6172676572205468616E204F6E6520426C6F636B2D53697A652044617461",
         "6F630FAD67CDA0EE1FB1F562DB3AA53E"},
        // RFC - 2202 HMAC-SHA1 test case 1
        {"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B",
         "4869205468657265",
         "B617318655057264E28BC0B6FB378C8EF146BE00"},
        // RFC - 2202 HMAC-SHA1 test case 2
        {"4A656665",
         "7768617420646F2079612077616E7420666F72206E6F7468696E673F",
         "EFFCDF6AE5EB2FA2D27416D5F184DF9C259A7C79"},
        // RFC - 2202 HMAC-SHA1 test case 3
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD",
         "125D7342B9AC11CD91A39AF48AA17B4F63F175D3"},
        // RFC - 2202 HMAC-SHA1 test case 4
        {"0102030405060708090A0B0C0D0E0F10111213141516171819",
         "CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD",
         "4C9007F4026250C6BC8414F9BF50C86C2D7235DA"},
        // RFC - 2202 HMAC-SHA1 test case 5
        {"0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C0C",
         "546573742057697468205472756E636174696F6E",
         "4C1A03424B55E07FE7F27BE1D58BB9324A9A5A04"},
        // RFC - 2202 HMAC-SHA1 test case 6
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B6579202D2048617368204B6579204669727374",
         "AA4AE5E15272D00E95705637CE8A3B55ED402112"},
        // RFC - 2202 HMAC-SHA1 test case 7
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B657920616E64204C6172676572205468616E204F6E6520426C6F636B2D53697A652044617461",
         "E8E99D0F45237D786D6BBAA7965C7808BBFF1A91"},
        // RFC - 4231 HMAC-SHA-256 test case 1
        {"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B",
         "4869205468657265",
         "B0344C61D8DB38535CA8AFCEAF0BF12B881DC200C9833DA726E9376C2E32CFF7"},
        // RFC - 4231 HMAC-SHA-384 test case 1
        {"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B",
         "4869205468657265",
         "AFD03944D84895626B0825F4AB46907F15F9DADBE4101EC682AA034C7CEBC59CFAEA9EA9076EDE7F4AF152E8B2FA9CB6"},
        // RFC - 4231 HMAC-SHA-512 test case 1
        {"0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B0B",
         "4869205468657265",
         "87AA7CDEA5EF619D4FF0B4241A1D6CB02379F4E2CE4EC2787AD0B30545E17CDEDAA833B7D6B8A702038B274EAEA3F4E4BE9D914EEB61F1702E696C203A126854"},
        // RFC - 4231 HMAC-SHA-256 test case 2
        {"4A656665",
         "7768617420646F2079612077616E7420666F72206E6F7468696E673F",
         "5BDCC146BF60754E6A042426089575C75A003F089D2739839DEC58B964EC3843"},
        // RFC - 4231 HMAC-SHA-384 test case 2
        {"4A656665",
         "7768617420646F2079612077616E7420666F72206E6F7468696E673F",
         "AF45D2E376484031617F78D2B58A6B1B9C7EF464F5A01B47E42EC3736322445E8E2240CA5E69E2C78B3239ECFAB21649"},
        // RFC - 4231 HMAC-SHA-512 test case 2
        {"4A656665",
         "7768617420646F2079612077616E7420666F72206E6F7468696E673F",
         "164B7A7BFCF819E2E395FBE73B56E0A387BD64222E831FD610270CD7EA2505549758BF75C05A994A6D034F65F8F0E6FDCAEAB1A34D4A6B4B636E070A38BCE737"},
        // RFC - 4231 HMAC-SHA-256 test case 3
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD",
         "773EA91E36800E46854DB8EBD09181A72959098B3EF8C122D9635514CED565FE"},
        // RFC - 4231 HMAC-SHA-384 test case 3
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD",
         "88062608D3E6AD8A0AA2ACE014C8A86F0AA635D947AC9FEBE83EF4E55966144B2A5AB39DC13814B94E3AB6E101A34F27"},
        // RFC - 4231 HMAC-SHA-512 test case 3
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD",
         "FA73B0089D56A284EFB0F0756C890BE9B1B5DBDD8EE81A3655F83E33B2279D39BF3E848279A722C806B485A47E67C807B946A337BEE8942674278859E13292FB"},
        // RFC - 4231 HMAC-SHA-256 test case 4
        {"0102030405060708090A0B0C0D0E0F10111213141516171819",
         "CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD",
         "82558A389A443C0EA4CC819899F2083A85F0FAA3E578F8077A2E3FF46729665B"},
        // RFC - 4231 HMAC-SHA-384 test case 4
        {"0102030405060708090a0b0c0d0e0f10111213141516171819",
         "CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD",
         "3E8A69B7783C25851933AB6290AF6CA77A9981480850009CC5577C6E1F573B4E6801DD23C4A7D679CCF8A386C674CFFB"},
        // RFC - 4231 HMAC-SHA-512 test case 4
        {"0102030405060708090a0b0c0d0e0f10111213141516171819",
         "CDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCDCD",
         "B0BA465637458C6990E5A8C5F61D4AF7E576D97FF94B872DE76F8050361EE3DBA91CA5C11AA25EB4D679275CC5788063A5F19741120C4F2DE2ADEBEB10A298DD"},
        // RFC - 4231 HMAC-SHA-256 test case 6
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B6579202D2048617368204B6579204669727374",
         "60E431591EE0B67F0D8A26AACBF5B77F8E0BC6213728C5140546040F0EE37F54"},
        // RFC - 4231 HMAC-SHA-384 test case 6
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B6579202D2048617368204B6579204669727374",
         "4ECE084485813E9088D2C63A041BC5B44F9EF1012A2B588F3CD11F05033AC4C60C2EF6AB4030FE8296248DF163F44952"},
        // RFC - 4231 HMAC-SHA-512 test case 6
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "54657374205573696E67204C6172676572205468616E20426C6F636B2D53697A65204B6579202D2048617368204B6579204669727374",
         "80B24263C7C1A3EBB71493C1DD7BE8B49B46D1F41B4AEEC1121B013783F8F3526B56D037E05F2598BD0FD2215D6A1E5295E64F73F63F0AEC8B915A985D786598"},
        // RFC - 4231 HMAC-SHA-256 test case 7
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "5468697320697320612074657374207573696E672061206C6172676572207468616E20626C6F636B2D73697A65206B657920616E642061206C6172676572207468616E20626C6F636B2D73697A6520646174612E20546865206B6579206E6565647320746F20626520686173686564206265666F7265206265696E6720757365642062792074686520484D414320616C676F726974686D2E",
         "9B09FFA71B942FCB27635FBCD5B0E944BFDC63644F0713938A7F51535C3A35E2"},
        // RFC - 4231 HMAC-SHA-384 test case 7
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "5468697320697320612074657374207573696E672061206C6172676572207468616E20626C6F636B2D73697A65206B657920616E642061206C6172676572207468616E20626C6F636B2D73697A6520646174612E20546865206B6579206E6565647320746F20626520686173686564206265666F7265206265696E6720757365642062792074686520484D414320616C676F726974686D2E",
         "6617178E941F020D351E2F254E8FD32C602420FEB0B8FB9ADCCEBB82461E99C5A678CC31E799176D3860E6110C46523E"},
        // RFC - 4231 HMAC-SHA-512 test case 7
        {"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA",
         "5468697320697320612074657374207573696E672061206C6172676572207468616E20626C6F636B2D73697A65206B657920616E642061206C6172676572207468616E20626C6F636B2D73697A6520646174612E20546865206B6579206E6565647320746F20626520686173686564206265666F7265206265696E6720757365642062792074686520484D414320616C676F726974686D2E",
         "E37B6A775DC87DBAA4DFA9F96E5E3FFDDEBD71F8867289865DF5A32D20CDC944B6022CAC3C4982B10D5EEB55C3E4DE15134676FB6DE0446065C97440FA8C6A58"},
    };

    public SymmetricSignatureImplTest(String testName) {
        super(testName);
    }

    protected void setUp() throws Exception {
        super.setUp();
    }

    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * Test of sign/verify methods, of class SymmetricSignatureImpl with 3DES Key
     */
    public void testSignVerify3DES() {
        SymmetricKeyImpl desKey = new SymmetricKeyImpl(KeyBuilder.TYPE_DES, KeyBuilder.LENGTH_DES3_3KEY);
        desKey.setKey(Hex.decode(DES3_KEY), (short) 0);
        testSignVerify(desKey, MESSAGE_8, MESSAGE_15, DES3_MAC_8,
                DES3_MAC_15, DES3_MAC_8_IV, DES3_MAC_15_IV);
    }

    /**
     * Test of sign/verify methods, of class SymmetricSignatureImpl with 2-key 3DES
     */
    public void testSignVerify2x3DES() {
        SymmetricKeyImpl desKey = new SymmetricKeyImpl(KeyBuilder.TYPE_DES, KeyBuilder.LENGTH_DES3_2KEY);
        desKey.setKey(Hex.decode(DES3_KEY), (short) 0);

        Signature engine = Signature.getInstance(Signature.ALG_DES_MAC8_ISO9797_1_M2_ALG3, false);
        testEngineSignVerify(engine, desKey, null, Hex.decode(MESSAGE_15), Hex.decode(DES3_MAC_15[2]));
        
    }
    
    /**
     * Test of sign/verifys methods, of class SymmetricSignatureImpl with DES Key
     */
    public void testSignVerifyDES() {
        SymmetricKeyImpl desKey = new SymmetricKeyImpl(KeyBuilder.TYPE_DES, KeyBuilder.LENGTH_DES);
        desKey.setKey(Hex.decode(DES_KEY), (short) 0);
        testSignVerify(desKey, MESSAGE_8, MESSAGE_15, DES_MAC_8,
                DES_MAC_15, DES_MAC_8_IV, DES_MAC_15_IV);
    }

    /**
     * Test of sign/verifys methods, of class SymmetricSignatureImpl with AES Key
     */
    public void testSignVerifyAES() {
        SymmetricKeyImpl aesKey = new SymmetricKeyImpl(KeyBuilder.TYPE_AES, KeyBuilder.LENGTH_AES_128);
        aesKey.setKey(Hex.decode(AES_128_KEY), (short) 0);
        Signature engine = Signature.getInstance(Signature.ALG_AES_MAC_128_NOPAD, false);
        testEngineSignVerify(engine, aesKey, null, Hex.decode(MESSAGE_16), Hex.decode(AES_CBC_MAC));
    }

    /**
     * Test of sign/verifys methods, of class SymmetricSignatureImpl with HMAC Key
     */
    public void testSignVerifyHMAC() {
        for (int i = 0; i < HMAC_IDS.length; i++) {
            byte[] hmacKeyBytes = Hex.decode(HMAC_TESTS[i][0]);
            HMACKey hmacKey = (HMACKey)KeyBuilder.buildKey(KeyBuilder.TYPE_HMAC, (short)hmacKeyBytes.length, false);
            hmacKey.setKey(hmacKeyBytes, (short) 0, (short)hmacKeyBytes.length);
            Signature engine = Signature.getInstance(HMAC_IDS[i], false);
            testEngineSignVerify(engine, hmacKey, null, Hex.decode(HMAC_TESTS[i][1]), Hex.decode(HMAC_TESTS[i][2]));
        }
    }

    /**
     * Test of sign/verify methods, of class SymmetricSignatureImpl with specified key
     * and etalon data
     */
    public void testSignVerify(SymmetricKeyImpl desKey, String msg8, String msg15,
            String[] enc8, String[] enc15, String[] enc8IV, String[] enc15IV) {

        // test DES MAC CBC with IV={0,0,0,0,0,0,0,0}
        Signature engine = Signature.getInstance(Signature.ALG_DES_MAC8_NOPAD, false);
        testEngineSignVerify(engine, desKey, null, Hex.decode(msg8), Hex.decode(enc8[0]));

        engine = Signature.getInstance(Signature.ALG_DES_MAC8_ISO9797_M1, false);
        testEngineSignVerify(engine, desKey, null, Hex.decode(msg15), Hex.decode(enc15[0]));

        engine = Signature.getInstance(Signature.ALG_DES_MAC8_ISO9797_M2, false);
        testEngineSignVerify(engine, desKey, null, Hex.decode(msg15), Hex.decode(enc15[1]));
      
        // test DES MAC CBC with non-zero IV
        byte[] iv = Hex.decode(IV);
        engine = Signature.getInstance(Signature.ALG_DES_MAC8_NOPAD, false);
        testEngineSignVerify(engine, desKey, iv, Hex.decode(msg8), Hex.decode(enc8IV[0]));

        engine = Signature.getInstance(Signature.ALG_DES_MAC8_ISO9797_M1, false);
        testEngineSignVerify(engine, desKey, iv, Hex.decode(msg15), Hex.decode(enc15IV[0]));

        engine = Signature.getInstance(Signature.ALG_DES_MAC8_ISO9797_M2, false);
        testEngineSignVerify(engine, desKey, iv, Hex.decode(msg15), Hex.decode(enc15IV[1]));
    }

    /**
     * Test of sign/verify methods, of class SymmetricSignatureImpl with specified key, engine
     * and etalon data
     * @param engine test engine
     * @param key etalon key
     * @param iv IV if present
     * @param msg etalon msg
     * @param macEtalon etalon signature(mac)
     */
    public void testEngineSignVerify(Signature engine, Key key, byte[] iv, byte[] msg, byte[] macEtalon) {
        // sign
        if (iv == null) {
            engine.init(key, Signature.MODE_SIGN);
        } else {
            engine.init(key, Signature.MODE_SIGN, iv, (short) 0, (short) iv.length);
        }
        byte[] mac = new byte[macEtalon.length];
        //
        engine.sign(msg, (short) 0, (short) msg.length, mac, (short) 0);
        assertEquals(true, Arrays.areEqual(mac, macEtalon));
        // verify
        if (iv == null) {
            engine.init(key, Signature.MODE_VERIFY);
        } else {
            engine.init(key, Signature.MODE_VERIFY, iv, (short) 0, (short) iv.length);
        }
        assertEquals(true, engine.verify(msg, (short) 0, (short) msg.length, macEtalon,
                (short) 0, (short) macEtalon.length));

    }
}
