let pidActive = (() => {
  var _ref = _asyncToGenerator(function* (pid) {
    if (!pid || isNaN(pid)) return false;
    return process.platform === 'win32' ? pidActiveWindows(pid) : pidActiveUnix(pid);
  });

  return function pidActive(_x) {
    return _ref.apply(this, arguments);
  };
})();

let lockActive = (() => {
  var _ref2 = _asyncToGenerator(function* (path) {
    try {
      let file = yield readFile(path);
      let pid = parseInt(file.trim());
      let active = pidActive(pid);
      if (!active) debug(`stale pid ${path} ${pid}`);
      return active;
    } catch (err) {
      if (err.code !== 'ENOENT') throw err;
      return false;
    }
  });

  return function lockActive(_x2) {
    return _ref2.apply(this, arguments);
  };
})();

let getReadersFile = (() => {
  var _ref3 = _asyncToGenerator(function* (path) {
    try {
      let f = yield readFile(path + '.readers');
      return f.split('\n').map(function (r) {
        return parseInt(r);
      });
    } catch (err) {
      return [];
    }
  });

  return function getReadersFile(_x3) {
    return _ref3.apply(this, arguments);
  };
})();

let getActiveReaders = (() => {
  var _ref4 = _asyncToGenerator(function* (path, timeout, skipOwnPid = false) {
    yield lock(path + '.readers.lock', timeout);
    let readers = yield getReadersFile(path);
    let promises = readers.map(function (r) {
      return pidActive(r).then(function (active) {
        return active ? r : null;
      });
    });
    let activeReaders = yield Promise.all(promises);
    activeReaders = activeReaders.filter(function (r) {
      return r !== null;
    });
    if (activeReaders.length !== readers.length) {
      yield saveReaders(path, activeReaders);
    }
    yield unlock(path + '.readers.lock');
    return skipOwnPid ? activeReaders.filter(function (r) {
      return r !== process.pid;
    }) : activeReaders;
  });

  return function getActiveReaders(_x4, _x5) {
    return _ref4.apply(this, arguments);
  };
})();

let waitForReaders = (() => {
  var _ref5 = _asyncToGenerator(function* (path, timeout, skipOwnPid) {
    let readers = yield getActiveReaders(path, timeout, skipOwnPid);
    if (readers.length !== 0) {
      if (timeout <= 0) throw new Error(`${path} is locked with ${readers.length === 1 ? 'a reader' : 'readers'} active: ${readers.join(' ')}`);
      debug(`waiting for readers: ${readers.join(' ')} timeout=${timeout}`);
      yield wait(1000);
      yield waitForReaders(path, timeout - 1000, skipOwnPid);
    }
  });

  return function waitForReaders(_x6, _x7, _x8) {
    return _ref5.apply(this, arguments);
  };
})();

let unread = (() => {
  var _ref6 = _asyncToGenerator(function* (path, timeout = 60000) {
    yield lock(path + '.readers.lock', timeout);
    let readers = yield getReadersFile(path);
    if (readers.find(function (r) {
      return r === process.pid;
    })) {
      yield saveReaders(path, readers.filter(function (r) {
        return r !== process.pid;
      }));
    }
    yield unlock(path + '.readers.lock');
  });

  return function unread(_x9) {
    return _ref6.apply(this, arguments);
  };
})();

/**
 * check if active writer
 * @param path {string} - path of lockfile to use
 */
let hasWriter = (() => {
  var _ref9 = _asyncToGenerator(function* (p) {
    let pid;
    try {
      pid = yield readFile(path.join(p + '.writer', 'pid'));
    } catch (err) {
      if (err.code !== 'ENOENT') throw err;
    }
    if (!pid) return false;
    let active = yield pidActive(parseInt(pid));
    return active;
  });

  return function hasWriter(_x12) {
    return _ref9.apply(this, arguments);
  };
})();

let hasReaders = (() => {
  var _ref10 = _asyncToGenerator(function* (p, options = {}) {
    let timeout = options.timeout || 60000;
    let skipOwnPid = !!options.skipOwnPid;
    let readers = yield getActiveReaders(p, timeout, skipOwnPid);
    return readers.length !== 0;
  });

  return function hasReaders(_x13) {
    return _ref10.apply(this, arguments);
  };
})();

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

const fs = require('fs-extra');
const path = require('path');

let debug;
try {
  debug = require('debug')('rwlockfile');
} catch (err) {
  debug = () => {};
}

let locks = {};
let readers = {};

function pidActiveWindows(pid) {
  return new Promise((resolve, reject) => {
    const { spawn } = require('child_process');
    const p = spawn('tasklist', ['/fi', `PID eq ${pid}`]);
    p.on('close', code => {
      if (code !== 0) reject(new Error(`tasklist exited with code ${code}`));
    });
    p.stdout.on('data', stdout => {
      resolve(!stdout.includes('No tasks are running'));
    });
  });
}

function pidActiveUnix(pid) {
  try {
    // flow$ignore
    return process.kill(pid, 0);
  } catch (e) {
    return e.code === 'EPERM';
  }
}

function unlock(path) {
  return new Promise(resolve => fs.remove(path, resolve)).then(() => {
    delete locks[path];
  });
}

function wait(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function unlockSync(path) {
  try {
    fs.removeSync(path);
  } catch (err) {
    debug(err);
  }
  delete locks[path];
}

function lock(p, timeout) {
  let pidPath = path.join(p, 'pid');
  if (!fs.existsSync(path.dirname(p))) fs.mkdirpSync(path.dirname(p));
  return new Promise((resolve, reject) => {
    fs.mkdir(p, err => {
      if (!err) {
        locks[p] = 1;
        fs.writeFile(pidPath, process.pid.toString(), resolve);
        return;
      }
      if (err.code !== 'EEXIST') return reject(err);
      lockActive(pidPath).then(active => {
        if (!active) return unlock(p).then(resolve).catch(reject);
        if (timeout <= 0) throw new Error(`${p} is locked`);
        debug(`locking ${p} ${timeout / 1000}s...`);
        wait(1000).then(() => lock(p, timeout - 1000).then(resolve).catch(reject));
      }).catch(reject);
    });
  });
}

function readFile(path) {
  return new Promise((resolve, reject) => {
    fs.readFile(path, 'utf8', (err, body) => {
      if (err) return reject(err);
      resolve(body);
    });
  });
}

function writeFile(path, content) {
  return new Promise((resolve, reject) => {
    fs.writeFile(path, content, (err, body) => {
      if (err) return reject(err);
      resolve(body);
    });
  });
}

function getReadersFileSync(path) {
  try {
    let f = fs.readFileSync(path + '.readers', 'utf8');
    return f.split('\n').map(r => parseInt(r));
  } catch (err) {
    return [];
  }
}

const unlink = p => new Promise((resolve, reject) => fs.unlink(p, err => err ? reject(err) : resolve()));

function saveReaders(path, readers) {
  path += '.readers';
  if (readers.length === 0) {
    return unlink(path).catch(() => {});
  } else {
    return writeFile(path, readers.join('\n'));
  }
}

function saveReadersSync(path, readers) {
  path += '.readers';
  try {
    if (readers.length === 0) {
      fs.unlinkSync(path);
    } else {
      fs.writeFileSync(path, readers.join('\n'));
    }
  } catch (err) {}
}

function waitForWriter(path, timeout) {
  return hasWriter(path).then(active => {
    if (active) {
      if (timeout <= 0) throw new Error(`${path} is locked with an active writer`);
      debug(`waiting for writer: path=${path} timeout=${timeout}`);
      return wait(1000).then(() => waitForWriter(path, timeout - 1000));
    }
    return unlock(path);
  });
}

exports.unread = unread;

function unreadSync(path) {
  // TODO: potential lock issue here since not using .readers.lock
  let readers = getReadersFileSync(path);
  saveReadersSync(path, readers.filter(r => r !== process.pid));
}

/**
 * lock for writing
 * @param path {string} - path of lockfile to use
 * @param options {object}
 * @param [options.timeout=60000] {number} - Max time to wait for lockfile to be open
 * @param [options.skipOwnPid] {boolean} - Do not wait on own pid (to upgrade current process)
 * @returns {Promise}
 */
exports.write = (() => {
  var _ref7 = _asyncToGenerator(function* (path, options = {}) {
    let skipOwnPid = !!options.skipOwnPid;
    let timeout = options.timeout || 60000;
    debug(`write ${path}`);
    yield waitForReaders(path, timeout, skipOwnPid);
    yield lock(path + '.writer', timeout);
    return function () {
      return unlock(path + '.writer');
    };
  });

  return function (_x10) {
    return _ref7.apply(this, arguments);
  };
})();

/**
 * lock for reading
 * @param path {string} - path of lockfile to use
 * @param options {object}
 * @param [options.timeout=60000] {number} - Max time to wait for lockfile to be open
 * @returns {Promise}
 */
exports.read = (() => {
  var _ref8 = _asyncToGenerator(function* (path, options = {}) {
    let timeout = options.timeout || 60000;
    debug(`read ${path}`);
    yield waitForWriter(path, timeout);
    yield lock(path + '.readers.lock', timeout);
    let readersFile = yield getReadersFile(path);
    yield saveReaders(path, readersFile.concat([process.pid]));
    yield unlock(path + '.readers.lock');
    readers[path] = 1;
    return function () {
      return unread(path, timeout);
    };
  });

  return function (_x11) {
    return _ref8.apply(this, arguments);
  };
})();
exports.hasWriter = hasWriter;

exports.hasReaders = hasReaders;

exports.unreadSync = unreadSync;

exports.cleanup = function () {
  Object.keys(locks).forEach(unlockSync);
  Object.keys(readers).forEach(unreadSync);
};

process.once('exit', exports.cleanup);