"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var model_1 = require("../model");
var graphql_1 = require("graphql");
var directives_1 = require("../directives");
// TODO(ejoebstl): It would be good to have this Parser fill the directive field for types and models as well.
/**
 * Parses a datamodel given as DSL
 * to an internal representation, convenient for
 * working with.
 */
var Parser = /** @class */ (function () {
    function Parser() {
    }
    /**
     * Shorthand to parse the datamodel, given an SDL string.
     * @param schemaString The datamodel as SDL string.
     * @returns A list of types found in the datamodel.
     */
    Parser.prototype.parseFromSchemaString = function (schemaString) {
        var schema = graphql_1.parse(schemaString);
        return this.parseFromSchema(schema);
    };
    /**
     * Parses the datamodel from a graphql-js schema.
     * @param schema The graphql-js schema, representing the datamodel.
     * @returns A list of types found in the datamodel.
     */
    Parser.prototype.parseFromSchema = function (schema) {
        var types = this.parseObjectTypes(schema).concat(this.parseEnumTypes(schema));
        this.resolveRelations(types);
        // Sort types alphabetically
        types.sort(function (_a, _b) {
            var a = _a.name;
            var b = _b.name;
            return (a > b ? 1 : -1);
        });
        // That's it.
        // We could check our model here, if we wanted to.
        // Possible checks:
        // * Check if we still use strings for identifying types for non-scalar types
        // * Check if all double-sided relations are connected correctly
        // * Check for duplicate type names
        // * Check for conflicting relations
        return {
            types: types
        };
    };
    /**
     * Finds a directive on a field or type by name.
     * @param fieldOrType
     * @param name
     */
    Parser.prototype.getDirectiveByName = function (fieldOrType, name) {
        var directive = fieldOrType.directives.filter(function (x) { return x.name.value === name; });
        return directive.length > 0 ? directive[0] : null;
    };
    /**
     * Checks if a directive on a given field or type ecists
     * @param fieldOrType
     * @param name
     */
    Parser.prototype.hasDirective = function (fieldOrType, name) {
        return this.getDirectiveByName(fieldOrType, name) != null;
    };
    /**
     * Checks if the given field is unique.
     * @param field
     */
    Parser.prototype.isUniqe = function (field) {
        return this.hasDirective(field, directives_1.DirectiveKeys.isUnique);
    };
    /**
     * Gets a fields default value. If no default
     * value is given, returns null.
     * @param field
     */
    Parser.prototype.getDefaultValue = function (field) {
        var directive = this.getDirectiveByName(field, directives_1.DirectiveKeys.default);
        var args = directive === null ? [] : directive.arguments.filter(function (x) { return x.name.value === 'value'; });
        return args.length !== 0 ? args[0].value.value : null;
    };
    /**
     * Gets a fields relation name. If no relation
     * exists, returns null.
     * @param field
     */
    Parser.prototype.getRelationName = function (field) {
        var directive = this.getDirectiveByName(field, directives_1.DirectiveKeys.relation);
        if (directive && directive.arguments) {
            var nameArgument = directive.arguments.find(function (a) { return a.name.value === 'name'; });
            return nameArgument ? nameArgument.value.value : null;
        }
        return null;
    };
    /**
     * Parses a model field, respects all
     * known directives.
     * @param field
     */
    Parser.prototype.parseField = function (field) {
        var name = field.name.value;
        var kind = this.parseKind(field.type, null);
        var fieldType = this.parseType(field.type);
        var isId = this.isIdField(field);
        var isUnique = isId || this.isUniqe(field);
        var isReadOnly = isId || this.isReadOnly(field);
        var defaultValue = this.getDefaultValue(field);
        var relationName = this.getRelationName(field);
        return {
            name: name,
            type: fieldType,
            relationName: relationName,
            defaultValue: defaultValue,
            isUnique: isUnique,
            isList: kind === 'ListType',
            isRequired: kind === 'NonNullType',
            relatedField: null,
            isId: isId,
            isReadOnly: isReadOnly,
        };
    };
    //  public isEmbedded(type: any): boolean {
    //    return type.directives &&
    //      type.directives.length > 0 &&
    //      type.directives.some(d => d.name.value === 'embedded')
    //  }
    /**
     * Parases an object type.
     * @param type
     */
    Parser.prototype.parseObjectType = function (type) {
        var fields = [];
        for (var _i = 0, _a = type.fields; _i < _a.length; _i++) {
            var field = _a[_i];
            if (field.kind === 'FieldDefinition') {
                // Check for type, kind, name and directives.
                fields.push(this.parseField(field));
            }
        }
        var isEmbedded = this.isEmbedded(type);
        return {
            name: type.name.value,
            fields: fields,
            isEnum: false,
            isEmbedded: isEmbedded,
        };
    };
    /**
     * Parses all object types in the schema.
     * @param schema
     */
    Parser.prototype.parseObjectTypes = function (schema) {
        var objectTypes = [];
        for (var _i = 0, _a = schema.definitions; _i < _a.length; _i++) {
            var type = _a[_i];
            if (type.kind === 'ObjectTypeDefinition') {
                objectTypes.push(this.parseObjectType(type));
            }
        }
        return objectTypes;
    };
    /**
     * Parses all enum types in the schema.
     * @param schema
     */
    Parser.prototype.parseEnumTypes = function (schema) {
        var enumTypes = [];
        for (var _i = 0, _a = schema.definitions; _i < _a.length; _i++) {
            var type = _a[_i];
            if (type.kind === 'EnumTypeDefinition') {
                var values = [];
                for (var _b = 0, _c = type.values; _b < _c.length; _b++) {
                    var value = _c[_b];
                    if (value.kind === 'EnumValueDefinition') {
                        var name_1 = value.name.value;
                        // All props except name are ignored for enum defs.
                        values.push(new model_1.GQLScalarField(name_1, 'String', false));
                    }
                }
                enumTypes.push({
                    name: type.name.value,
                    fields: values,
                    isEnum: true,
                    isEmbedded: false,
                });
            }
        }
        return enumTypes;
    };
    /**
     * Resolves and connects all realtion fields found
     * in the given type list.
     * @param types
     */
    Parser.prototype.resolveRelations = function (types) {
        // Find all types that we know,
        // and assign a proper type object instead
        // of the string.
        for (var _i = 0, types_1 = types; _i < types_1.length; _i++) {
            var typeA = types_1[_i];
            for (var _a = 0, _b = typeA.fields; _a < _b.length; _a++) {
                var fieldA = _b[_a];
                for (var _c = 0, types_2 = types; _c < types_2.length; _c++) {
                    var typeB = types_2[_c];
                    // At this stage, every type is a string
                    if (fieldA.type === typeB.name) {
                        fieldA.type = typeB;
                    }
                }
            }
        }
        // Connect all relations that are named.
        for (var _d = 0, types_3 = types; _d < types_3.length; _d++) {
            var typeA = types_3[_d];
            for (var _e = 0, _f = typeA.fields; _e < _f.length; _e++) {
                var fieldA = _f[_e];
                if (typeof fieldA.type === 'string') {
                    continue; // Assume scalar
                }
                if (fieldA.relationName !== null && fieldA.relatedField === null) {
                    for (var _g = 0, _h = fieldA.type.fields; _g < _h.length; _g++) {
                        var fieldB = _h[_g];
                        if (fieldB.relationName === fieldA.relationName) {
                            if (fieldB.type !== typeA) {
                                throw new Error("Relation type mismatch for relation " + fieldA.relationName);
                            }
                            fieldA.relatedField = fieldB;
                            fieldB.relatedField = fieldA;
                            break;
                        }
                    }
                }
            }
        }
        // Connect  obvious relations which are lacking the relatioName directive.
        // We explicitely DO NOT ignore fields with a given relationName, in accordance
        // to the prisma implementation.
        for (var _j = 0, types_4 = types; _j < types_4.length; _j++) {
            var typeA = types_4[_j];
            searchThroughAFields: for (var _k = 0, _l = typeA.fields; _k < _l.length; _k++) {
                var fieldA = _l[_k];
                if (typeof fieldA.type === 'string') {
                    continue; // Assume scalar.
                }
                if (fieldA.relatedField !== null) {
                    continue; // Nothing to do, already connected
                }
                for (var _m = 0, _o = typeA.fields; _m < _o.length; _m++) {
                    var fieldA2 = _o[_m];
                    if (fieldA2 !== fieldA && fieldA2.type === fieldA.type) {
                        // Skip, A has more than one fields of this relation type.
                        continue searchThroughAFields;
                    }
                }
                var relationPairs = [];
                // Look for the opposite field by type.
                for (var _p = 0, _q = fieldA.type.fields; _p < _q.length; _p++) {
                    var fieldB = _q[_p];
                    if (fieldB.type === typeA) {
                        if (fieldB !== fieldA) {
                            // Don't connect self-referencing fields
                            relationPairs.push({ a: fieldA, b: fieldB });
                        }
                    }
                }
                // Create relation iff we have found a single pair
                if (relationPairs.length === 1) {
                    var _r = relationPairs[0], a = _r.a, b = _r.b;
                    a.relatedField = b;
                    b.relatedField = a;
                }
            }
        }
    };
    /**
     * Traverses an AST branch and finds the next type.
     * This will skip modifiers like NonNullType or ListType.
     * @param type
     */
    Parser.prototype.parseType = function (type) {
        if (type.type) {
            return this.parseType(type.type);
        }
        else if (type.kind !== 'NamedType') {
            throw new Error();
        }
        return type.name.value;
    };
    /**
     * Traverses an AST branch and returns the modifier
     * of the type: Either ListType or NonNullType.
     * @param type
     * @param acc
     */
    Parser.prototype.parseKind = function (type, acc) {
        if (!acc) {
            acc = type.kind;
        }
        // If we find list, we always take list
        if (type.kind === 'ListType') {
            return type.kind;
        }
        // Non-null has higher prio than nullable
        if (type.kind === 'NonNullType') {
            acc = type.kind;
        }
        // When we reach the end, return whatever we have stored.
        if (type.type) {
            return this.parseKind(type.type, acc);
        }
        else {
            return acc;
        }
    };
    return Parser;
}());
exports.default = Parser;
//# sourceMappingURL=parser.js.map