/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 * @format
 */

"use strict";
var _extends =
  Object.assign ||
  function(target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];
      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }
    return target;
  };
function _asyncToGenerator(fn) {
  return function() {
    var gen = fn.apply(this, arguments);
    return new Promise(function(resolve, reject) {
      function step(key, arg) {
        try {
          var info = gen[key](arg);
          var value = info.value;
        } catch (error) {
          reject(error);
          return;
        }
        if (info.done) {
          resolve(value);
        } else {
          return Promise.resolve(value).then(
            function(value) {
              step("next", value);
            },
            function(err) {
              step("throw", err);
            }
          );
        }
      }
      return step("next");
    });
  };
}

const Bundler = require("./Bundler");
const DeltaBundler = require("./DeltaBundler");
const ResourceNotFoundError = require("./IncrementalBundler/ResourceNotFoundError");

const crypto = require("crypto");
const fs = require("fs");
const getGraphId = require("./lib/getGraphId");
const getPrependedScripts = require("./lib/getPrependedScripts");
const path = require("path");
const transformHelpers = require("./lib/transformHelpers");

function createRevisionId() {
  return crypto.randomBytes(8).toString("hex");
}

function revisionIdFromString(str) {
  return str;
}

class IncrementalBundler {
  constructor(config) {
    this._revisionsById = new Map();
    this._revisionsByGraphId = new Map();
    this._config = config;
    this._bundler = new Bundler(config);
    this._deltaBundler = new DeltaBundler(this._bundler);
  }

  end() {
    this._deltaBundler.end();
    this._bundler.end();
  }

  getBundler() {
    return this._bundler;
  }

  getDeltaBundler() {
    return this._deltaBundler;
  }

  getRevision(revisionId) {
    return this._revisionsById.get(revisionId);
  }

  getRevisionByGraphId(graphId) {
    return this._revisionsByGraphId.get(graphId);
  }

  buildGraphForEntries(entryFiles, transformOptions) {
    var _this = this;
    let otherOptions =
      arguments.length > 2 && arguments[2] !== undefined
        ? arguments[2]
        : { onProgress: null };
    return _asyncToGenerator(function*() {
      const absoluteEntryFiles = entryFiles.map(function(entryFile) {
        return path.resolve(_this._config.projectRoot, entryFile);
      });

      yield Promise.all(
        absoluteEntryFiles.map(function(entryFile) {
          return new Promise(function(resolve, reject) {
            // This should throw an error if the file doesn't exist.
            // Using this instead of fs.exists to account for SimLinks.
            fs.realpath(entryFile, function(err) {
              if (err) {
                reject(new ResourceNotFoundError(entryFile));
              } else {
                resolve();
              }
            });
          });
        })
      );

      const graph = yield _this._deltaBundler.buildGraph(absoluteEntryFiles, {
        resolve: yield transformHelpers.getResolveDependencyFn(
          _this._bundler,
          transformOptions.platform
        ),

        transform: yield transformHelpers.getTransformFn(
          absoluteEntryFiles,
          _this._bundler,
          _this._deltaBundler,
          _this._config,
          transformOptions
        ),

        onProgress: otherOptions.onProgress
      });

      _this._config.serializer.experimentalSerializerHook(graph, {
        modified: graph.dependencies,
        deleted: new Set(),
        reset: true
      });

      return graph;
    })();
  }

  buildGraph(entryFile, transformOptions) {
    var _this2 = this;
    let otherOptions =
      arguments.length > 2 && arguments[2] !== undefined
        ? arguments[2]
        : { onProgress: null };
    return _asyncToGenerator(function*() {
      const graph = yield _this2.buildGraphForEntries(
        [entryFile],
        transformOptions,
        otherOptions
      );

      const transformOptionsWithoutType = {
        customTransformOptions: transformOptions.customTransformOptions,
        dev: transformOptions.dev,
        experimentalImportSupport: transformOptions.experimentalImportSupport,
        hot: transformOptions.hot,
        minify: transformOptions.minify,
        platform: transformOptions.platform
      };

      const prepend = yield getPrependedScripts(
        _this2._config,
        transformOptionsWithoutType,
        _this2._bundler,
        _this2._deltaBundler
      );

      return {
        prepend,
        graph
      };
    })();
  }

  // TODO T34760750 (alexkirsz) Eventually, I'd like to get to a point where
  // this class exposes only initializeGraph and updateGraph.
  initializeGraph(entryFile, transformOptions) {
    var _this3 = this;
    let otherOptions =
      arguments.length > 2 && arguments[2] !== undefined
        ? arguments[2]
        : { onProgress: null };
    return _asyncToGenerator(function*() {
      const graphId = getGraphId(entryFile, transformOptions);
      const revisionId = createRevisionId();
      const revisionPromise = _asyncToGenerator(function*() {
        var _ref2 = yield _this3.buildGraph(
          entryFile,
          transformOptions,
          otherOptions
        );
        const graph = _ref2.graph,
          prepend = _ref2.prepend;

        return {
          id: revisionId,
          date: new Date(),
          graphId,
          graph,
          prepend
        };
      })();

      _this3._revisionsById.set(revisionId, revisionPromise);
      _this3._revisionsByGraphId.set(graphId, revisionPromise);
      const revision = yield revisionPromise;

      const delta = {
        modified: revision.graph.dependencies,
        deleted: new Set(),
        reset: true
      };

      return {
        revision,
        delta
      };
    })();
  }

  updateGraph(revision, reset) {
    var _this4 = this;
    return _asyncToGenerator(function*() {
      const delta = yield _this4._deltaBundler.getDelta(revision.graph, {
        reset
      });

      _this4._config.serializer.experimentalSerializerHook(
        revision.graph,
        delta
      );

      if (delta.modified.size > 0) {
        _this4._revisionsById.delete(revision.id);
        revision = _extends({}, revision, {
          // Generate a new revision id, to be used to verify the next incremental
          // request.
          id: crypto.randomBytes(8).toString("hex"),
          date: new Date()
        });

        const revisionPromise = Promise.resolve(revision);
        _this4._revisionsById.set(revision.id, revisionPromise);
        _this4._revisionsByGraphId.set(revision.graphId, revisionPromise);
      }

      return { revision, delta };
    })();
  }
}
IncrementalBundler.revisionIdFromString = revisionIdFromString;

module.exports = IncrementalBundler;
