"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var _ = require("lodash");
var SdlPrinter = /** @class */ (function () {
    function SdlPrinter() {
    }
    SdlPrinter.prototype.print = function (tables) {
        var _this = this;
        var candidates = tables.filter(function (x) { return !x.isJunctionTable; });
        var sdl = _.map(candidates, function (table) {
            return _this.printType(table, tables.filter(function (x) { return x != table; }));
        });
        return sdl.join('\r\n');
    };
    SdlPrinter.prototype.printType = function (table, otherTables) {
        var _this = this;
        var nativeFields = table.columns;
        var filterFunction = function (c) { return (c.relation || false) && c.relation.table == table.name; };
        var relationFields = otherTables
            .filter(function (t) { return t.columns.some(filterFunction); })
            .map(function (t) {
            return t.columns
                .filter(filterFunction)
                .map(function (c) { return ({ remoteColumn: c, remoteTable: t }); });
        })
            .reduce(function (acc, next) { return acc.concat(next); }, []);
        return "type " + this.capitalizeFirstLetter(table.name) + " @pgTable(name: \"" + table.name + "\") {\n  " + (_.map(nativeFields, function (nativeField) { return _this.printField(nativeField); }).concat(relationFields.map(function (field) { return _this.printBackRelationField(field); }))).join('\n  ') + "\n}\n";
    };
    SdlPrinter.prototype.printBackRelationField = function (field) {
        if (field.remoteTable.isJunctionTable) {
            var otherRemoteTableField = field.remoteTable.columns.filter(function (x) { return x.name !== field.remoteColumn.name; })[0];
            var relatedTable = otherRemoteTableField.relation.table;
            return this.lowerCaseFirstLetter(relatedTable) + "s: [" + this.capitalizeFirstLetter(relatedTable) + "!]! @pgRelationTable(table: \"" + field.remoteTable.name + "\" name: \"" + field.remoteTable.name + "\")";
        }
        else {
            return field.remoteTable.name + "s: [" + this.capitalizeFirstLetter(field.remoteTable.name) + "!]!";
        }
    };
    SdlPrinter.prototype.printField = function (column) {
        var field = this.printFieldName(column) + ": " + this.printFieldType(column) + this.printFieldOptional(column) + this.printRelationDirective(column) + this.printFieldDirectives(column) + (column.comment === null ? "" : " # " + column.comment);
        if (column.typeIdentifier === null) {
            return "# " + field;
        }
        return field;
    };
    SdlPrinter.prototype.printFieldName = function (column) {
        if (column.relation) {
            return this.removeIdSuffix(column.name);
        }
        else if (column.isPrimaryKey) {
            return "id";
        }
        else {
            return column.name;
        }
    };
    SdlPrinter.prototype.printFieldType = function (column) {
        if (column.relation) {
            return this.capitalizeFirstLetter(column.relation.table);
        }
        else {
            return column.typeIdentifier;
        }
    };
    SdlPrinter.prototype.printRelationDirective = function (column) {
        if (column.relation) {
            return " @pgRelation(column: \"" + column.name + "\")";
        }
        else {
            return '';
        }
    };
    SdlPrinter.prototype.printFieldOptional = function (column) {
        return column.nullable ? '' : '!';
    };
    SdlPrinter.prototype.printFieldDirectives = function (column) {
        var directives = '';
        if (column.isUnique) {
            directives += " @unique";
        }
        if (column.isPrimaryKey && column.name != "id") {
            directives += " @pgColumn(name: \"" + column.name + "\")";
        }
        if (column.defaultValue != null) {
            if (column.defaultValue == '[AUTO INCREMENT]') {
                return directives;
            }
            if (column.typeIdentifier == 'String' ||
                column.typeIdentifier == 'DateTime' ||
                column.typeIdentifier == 'Json') {
                directives += " @default(value: \"" + column.defaultValue + "\")";
            }
            else {
                directives += " @default(value: " + column.defaultValue + ")";
            }
        }
        return directives;
    };
    SdlPrinter.prototype.capitalizeFirstLetter = function (string) {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };
    SdlPrinter.prototype.lowerCaseFirstLetter = function (string) {
        return string.charAt(0).toLowerCase() + string.slice(1);
    };
    SdlPrinter.prototype.removeIdSuffix = function (string) {
        function removeSuffix(suffix, string) {
            if (string.endsWith(suffix)) {
                return string.substring(0, string.length - suffix.length);
            }
            else {
                return string;
            }
        }
        return removeSuffix('_ID', removeSuffix('_id', removeSuffix('Id', string)));
    };
    return SdlPrinter;
}());
exports.SdlPrinter = SdlPrinter;
//# sourceMappingURL=SdlPrinter.js.map