import DOMMatrix from './DOMMatrix'
import {applyArrayValuesToDOMMatrix} from './utilities'

// This matrix is represented internally in row-major format so that it is easy
// to look at visually. In a pair of coordinates (as in "m23") the first number
// is the column and the second is the row (so "m23" means column 2 row 3).
const identity = [
    /*m11*/1, /*m21*/0, /*m31*/0, /*m41*/0,
    /*m12*/0, /*m22*/1, /*m32*/0, /*m42*/0,
    /*m13*/0, /*m23*/0, /*m33*/1, /*m43*/0,
    /*m14*/0, /*m24*/0, /*m34*/0, /*m44*/1 ]

var DOMMatrixReadOnly = null

export
function initDOMMatrixReadOnly() {
    if (DOMMatrixReadOnly) { return }

    DOMMatrixReadOnly = (function () {
        function DOMMatrixReadOnly(numberSequence) {
        if ( numberSequence === void 0 ) numberSequence = [];

            if (!(this instanceof DOMMatrix))
                { throw new TypeError("DOMMatrixReadOnly can't be instantiated directly. Use DOMMatrix instead.") }

            var length = numberSequence.length;

            if (length === undefined || !(length === 6 || length === 16))
                { throw new TypeError('DOMMatrix constructor argument "numberSequence" must be an array-like with 6 or 16 numbers.') }

            this._matrix = new Float64Array(identity)
            this._isIdentity = true
            this._is2D = length === 6 ? true : false

            applyArrayValuesToDOMMatrix(numberSequence, this)
        }

        var prototypeAccessors = { is2D: {},isIdentity: {},a: {},b: {},c: {},d: {},e: {},f: {},m11: {},m12: {},m13: {},m14: {},m21: {},m22: {},m23: {},m24: {},m31: {},m32: {},m33: {},m34: {},m41: {},m42: {},m43: {},m44: {} };

        // Immutable transform methods -------------------------------------------

        DOMMatrixReadOnly.prototype.translate = function translate (tx, ty, tz) {
            if ( tz === void 0 ) tz = 0;

            return new DOMMatrix(this).translateSelf(tx, ty, tz)
        };

        DOMMatrixReadOnly.prototype.scale = function scale (scale$1, originX, originY) {
            if ( originX === void 0 ) originX = 0;
            if ( originY === void 0 ) originY = 0;

            return new DOMMatrix(this).scaleSelf(scale$1, originX, originY)
        };

        DOMMatrixReadOnly.prototype.scale3d = function scale3d (scale, originX, originY, originZ) {
            if ( originX === void 0 ) originX = 0;
            if ( originY === void 0 ) originY = 0;
            if ( originZ === void 0 ) originZ = 0;

            return new DOMMatrix(this).scale3dSelf(scale, originX, originY, originZ)
        };

        DOMMatrixReadOnly.prototype.scaleNonUniform = function scaleNonUniform (scaleX, scaleY, scaleZ, originX, originY, originZ) {
            if ( scaleY === void 0 ) scaleY = 1;
            if ( scaleZ === void 0 ) scaleZ = 1;
            if ( originX === void 0 ) originX = 0;
            if ( originY === void 0 ) originY = 0;
            if ( originZ === void 0 ) originZ = 0;

            return new DOMMatrix(this).scaleNonUniformSelf(scaleX, scaleY, scaleZ, originX, originY, originZ)
        };

        DOMMatrixReadOnly.prototype.rotate = function rotate (angle, originX, originY) {
            if ( originX === void 0 ) originX = 0;
            if ( originY === void 0 ) originY = 0;

            return new DOMMatrix(this).rotateSelf(angle, originX, originY)
        };

        // TODO
        DOMMatrixReadOnly.prototype.rotateFromVector = function rotateFromVector (x, y) {
            throw new Error('rotateFromVector is not implemented yet.')
        };

        DOMMatrixReadOnly.prototype.rotateAxisAngle = function rotateAxisAngle (x, y, z, angle) {
            return new DOMMatrix(this).rotateAxisAngleSelf(x, y, z, angle)
        };

        DOMMatrixReadOnly.prototype.skewX = function skewX (sx) {
            throw new Error('skewX is not implemented yet.')
        };
        DOMMatrixReadOnly.prototype.skewY = function skewY (sy) {
            throw new Error('skewY is not implemented yet.')
        };

        DOMMatrixReadOnly.prototype.multiply = function multiply (other) {
            return new DOMMatrix(this).multiplySelf(other)
        };

        DOMMatrixReadOnly.prototype.flipX = function flipX () {
            throw new Error('flipX is not implemented yet.')
        };
        DOMMatrixReadOnly.prototype.flipY = function flipY () {
            throw new Error('flipY is not implemented yet.')
        };
        DOMMatrixReadOnly.prototype.inverse = function inverse () {
            throw new Error('inverse is not implemented yet.')
        };

        DOMMatrixReadOnly.prototype.transformPoint = function transformPoint (/*optional DOMPointInit*/ point) {
            throw new Error('transformPoint is not implemented yet.')
        };

        DOMMatrixReadOnly.prototype.toFloat32Array = function toFloat32Array () {
            return Float32Array.from(this._matrix)
        };
        DOMMatrixReadOnly.prototype.toFloat64Array = function toFloat64Array () {
            return Float64Array.from(this._matrix)
        };

        //stringifier() {} // What's this?

        prototypeAccessors.is2D.get = function () {
            return this._is2D
        };

        /*
         * TODO: make sure this matches the spec.
         * TODO: Instead of calculating here, perhaps calculate and set
         * this._isIdentity in other operations, and simply return the internal one
         * here.
         */
        prototypeAccessors.isIdentity.get = function () {
            var this$1 = this;

            for (var i = 0, len = this._matrix.length; i < len; i+=1) {
                if (this$1._matrix[i] != identity[i])
                    { return (this$1._isIdentity = false) }
            }

            return (this._isIdentity = true)
        };

        prototypeAccessors.a.get = function () { return this.m11 };
        prototypeAccessors.b.get = function () { return this.m12 };
        prototypeAccessors.c.get = function () { return this.m21 };
        prototypeAccessors.d.get = function () { return this.m22 };
        prototypeAccessors.e.get = function () { return this.m41 };
        prototypeAccessors.f.get = function () { return this.m42 };

        prototypeAccessors.m11.get = function () { return this._matrix[0]  };
        prototypeAccessors.m12.get = function () { return this._matrix[4]  };
        prototypeAccessors.m13.get = function () { return this._matrix[8]  };
        prototypeAccessors.m14.get = function () { return this._matrix[12] };

        prototypeAccessors.m21.get = function () { return this._matrix[1]  };
        prototypeAccessors.m22.get = function () { return this._matrix[5]  };
        prototypeAccessors.m23.get = function () { return this._matrix[9]  };
        prototypeAccessors.m24.get = function () { return this._matrix[13] };

        prototypeAccessors.m31.get = function () { return this._matrix[2]  };
        prototypeAccessors.m32.get = function () { return this._matrix[6]  };
        prototypeAccessors.m33.get = function () { return this._matrix[10] };
        prototypeAccessors.m34.get = function () { return this._matrix[14] };

        prototypeAccessors.m41.get = function () { return this._matrix[3]  };
        prototypeAccessors.m42.get = function () { return this._matrix[7]  };
        prototypeAccessors.m43.get = function () { return this._matrix[11] };
        prototypeAccessors.m44.get = function () { return this._matrix[15] };

        Object.defineProperties( DOMMatrixReadOnly.prototype, prototypeAccessors );

        return DOMMatrixReadOnly;
    }())
}

initDOMMatrixReadOnly()

// export default as a "live binding", not a static default.
export {DOMMatrixReadOnly as default}

//# sourceMappingURL=data:application/json;charset=utf-8;base64,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