package de.csicar.mensaplan.kitcard;

/*
 * This file is taken and modified from KITCard Reader. (https://github.com/pkern/kitcard-reader)
 * Ⓒ 2012 Philipp Kern <phil@philkern.de>
 *
 * KITCard Reader is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * KITCard Reader is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with KITCard Reader. If not, see <http://www.gnu.org/licenses/>.
 */

import android.nfc.tech.MifareClassic;
import android.util.Log;

import org.joda.money.CurrencyUnit;
import org.joda.money.Money;
import org.joda.money.format.MoneyAmountStyle;
import org.joda.money.format.MoneyFormatter;
import org.joda.money.format.MoneyFormatterBuilder;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;

import de.csicar.mensaplan.CardType;

/**
 * Wallet: Read a MifareClassic tag, assuming a KITCard wallet.
 *
 * The first 3 blocks of sector 6 are accessed read-only. They contain
 * the current balance, the previous balance (the blocks are written in
 * an alternating fashion), transaction counters, a card type, and some
 * crypto keys.
 *
 * The card number can also be queried. It is stored as a string in the
 * first 12 bytes of sector 11.
 *
 * Based on reverse engineering work by Fabian Knittel <fabian@lettink.de>.
 *
 * @author Philipp Kern <pkern@debian.org>
 */
public class Wallet {
    enum ReadCardResult {
        SUCCESS,
        FAILURE,
        OLD_STYLE_WALLET;
    }

    private static final String LOG_TAG = "KITCard Reader";
    private static final byte[] CARD_NUMBER_KEY = {(byte)0x56, (byte)0x38, (byte)0x9f, (byte)0x80, (byte)0xa5, (byte)0xcf};
    private static final byte[] WALLET_KEY = MifareClassic.KEY_MIFARE_APPLICATION_DIRECTORY;
    private static final int WALLET_TCOUNT_KEY = 0x0404;
    private static final int WALLET_FCC = 0x17;
    private static final int WALLET_AC = 0x89;
    private static final int WALLET_OLD_FCC = 0x3;
    private static final int WALLET_OLD_AC = 0x38;

    private static final CurrencyUnit currency = CurrencyUnit.EUR;
    private static final int moneyDecimalPointOffset = 2;
    private static final MoneyFormatter moneyFormatter =
            new MoneyFormatterBuilder()
            .appendAmount(MoneyAmountStyle.ASCII_DECIMAL_COMMA_GROUP3_DOT)
            .appendCurrencySymbolLocalized()
            .toFormatter(Locale.GERMANY);



    private final MifareClassic card;
    private String cardNumber;
    private Money currentBalance;
    private Money lastBalance;
    private CardType cardType;
    private int transactionCount1;

    private int transactionCount2;

    public Wallet(MifareClassic card) {
        this.card = card;
    }

    public ReadCardResult readCard() {
        try {
            try {
                if(!card.isConnected()) {
                    card.connect();
                    Log.d(LOG_TAG, "Connect to tag successful");
                    if(card.authenticateSectorWithKeyA(11, CARD_NUMBER_KEY)) {
                        parseCardNumber(card.readBlock(card.sectorToBlock(11)));
                    } else {
                        Log.e(LOG_TAG, "Authentication with sector 11 failed");
                    }
                    final MifareMad mad = new MifareMad(card);
                    final ArrayList<Integer> sectorList = mad.getSectorList(WALLET_FCC, WALLET_AC);
                    if(sectorList.size() != 1) {
                        // Either not a card with a KITCard wallet or an
                        // old-style card, let's check.
                        if(mad.getSectorList(WALLET_OLD_FCC, WALLET_OLD_AC).size() == 2) {
                            Log.e(LOG_TAG, "Old-style wallet found, needs reencoding.");
                            return ReadCardResult.OLD_STYLE_WALLET;
                        }
                        Log.e(LOG_TAG, "No wallet data found (neither new- nor old-style), not a KITCard?");
                        return ReadCardResult.FAILURE;
                    }
                    int sector = sectorList.get(0);
                    if(!card.authenticateSectorWithKeyA(sector, WALLET_KEY)) {
                        Log.e(LOG_TAG, String.format("Authentication with sector %d (wallet) failed", sector));
                        return ReadCardResult.FAILURE;
                    }
                    parseWalletData(
                            card.readBlock(card.sectorToBlock(sector)),
                            card.readBlock(card.sectorToBlock(sector) + 1),
                            card.readBlock(card.sectorToBlock(sector) + 2)
                    );
                }
            } finally {
                card.close();
            }
            return ReadCardResult.SUCCESS;
        } catch(IOException e) {
            Log.e(LOG_TAG, "IOException caught: " + e.toString());
            return ReadCardResult.FAILURE;
        }
    }

    private static String bytesToString(byte[] ary) {
        final StringBuilder result = new StringBuilder();
        for(int i = 0; i < ary.length; ++i) {
            result.append(Character.valueOf((char)ary[i]));
        }
        return result.toString();
    }

    private void parseCardNumber(byte[] data) {
        final byte[] number = Arrays.copyOfRange(data, 0, 12);
        cardNumber = bytesToString(number);
    }

    /**
     * Logs the content of a byte array as hex at debug level.
     *
     * @param block The block to print.
     */
    private static void debugPrintBlock(byte[] block) {
        final StringBuilder builder = new StringBuilder("hex ");
        for(byte b : block) {
            final String hexPart = Integer.toHexString(MifareUtils.byteToInt(b)).toUpperCase();
            if(hexPart.length() < 2)
                builder.append("0");
            builder.append(hexPart);
            builder.append(" ");
        }

        Log.d(LOG_TAG, builder.toString());
    }

    private void parseWalletData(byte[] block1, byte[] block2, byte[] block3) {
        final byte[] data = new byte[16*3];
        System.arraycopy(block1, 0, data, 0, 16);
        System.arraycopy(block2, 0, data, 16, 16);
        System.arraycopy(block3, 0, data, 32, 16);
        debugPrintBlock(block1);
        debugPrintBlock(block2);
        debugPrintBlock(block3);

        // Decrypt in-place.
        int data_key = MifareUtils.byteToInt(data[41]);
        for(int i = 9; i <= 44; ++i) {
            data[i] = MifareUtils.intToByte(MifareUtils.byteToInt(data[i]) ^ data_key);
        }

        int value_key = MifareUtils.toUInt16BE(data[42], data[43]);
        int front_value = MifareUtils.toUInt16LE(data[26], data[27]) ^ value_key;
        int back_value = MifareUtils.toUInt16LE(data[31], data[32]) ^ value_key ^ 0x3b05;
        int front_count = MifareUtils.toUInt16BE(data[28], data[29]) ^ WALLET_TCOUNT_KEY;
        int back_count = MifareUtils.toUInt16BE(data[33], data[34]) ^ WALLET_TCOUNT_KEY ^ 0x3d3e;

        Log.d(LOG_TAG, String.format("Front count: %d; Back count: %d", front_count, back_count));

        if(front_count == back_count) {
            currentBalance = convertIntToMoney(front_value);
            lastBalance = convertIntToMoney(back_value);
        } else {
            currentBalance = convertIntToMoney(back_value);
            lastBalance = convertIntToMoney(front_value);
        }

        int status = MifareUtils.toUInt16LE(data[38], data[39]) ^ 0x3f3e;
        Log.d(LOG_TAG, "Status: " + Integer.toString(status));
        switch(status) {
            case 100:
                cardType = CardType.STUDENT;
                break;
            case 400: // UB
                cardType = CardType.EMPLOYEE;
                break;
            case 403: // GFB
                cardType = CardType.EMPLOYEE;
                break;
            case 200:
                cardType = CardType.GUEST;
                break;
            case 201:
                cardType = CardType.GUEST;
                break;
            default:
                cardType = CardType.UNKNOWN;
                break;
        }
    }

    private static Money convertIntToMoney(int moneyInt) {
        return Money.of(currency, BigDecimal.valueOf(moneyInt, moneyDecimalPointOffset));
    }

    public String getCardNumber() {
        return cardNumber;
    }

    public String getCardIssuer() {
        if (cardNumber == null) {
            return "HS Pforzheim";
        } else if (cardNumber.startsWith("1580")) {
            return "KIT";
        } else if (cardNumber.startsWith("6760")) {
            return "HS Karlsruhe";
        } else if (cardNumber.startsWith("3680")) {
            return "PH Karlsruhe";
        } else if (cardNumber.startsWith("146")) { // HS-Schlüssel: 8175; legacy
            return "DHBW Karlsruhe";
        } else {
            return cardNumber.substring(0, 4) ;
        }
    }

    public Money getCurrentBalance() {
        return currentBalance;
    }

    public Money getLastBalance() {
        return lastBalance;
    }

    public Money getLastTransactionValue() {
        return currentBalance.minus(lastBalance);
    }

    public CardType getCardType() {
        return cardType;
    }

    public String getCurrentBalanceText() {
        return moneyFormatter.print(currentBalance);
    }

    public String getLastBalanceText() {
        return moneyFormatter.print(lastBalance);
    }

    public String getLastTransactionText() {
        return moneyFormatter.print(getLastTransactionValue());
    }

    public int getTransactionCount1() {
        return transactionCount1;
    }

    public int getTransactionCount2() {
        return transactionCount2;
    }

}