package com.vonglasow.michael.qz.core;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.regex.Pattern;

import eu.jacquet80.rds.app.oda.AlertC;
import eu.jacquet80.rds.app.oda.AlertC.InformationBlock;
import eu.jacquet80.rds.app.oda.AlertC.Message;
import eu.jacquet80.rds.app.oda.tmc.Segment;
import eu.jacquet80.rds.app.oda.tmc.TMC;
import eu.jacquet80.rds.app.oda.tmc.TMCEvent.EventUrgency;
import eu.jacquet80.rds.app.oda.tmc.TMCLocation;
import eu.jacquet80.rds.app.oda.tmc.TMCLocation.LocationClass;
import eu.jacquet80.rds.app.oda.tmc.TMCPoint;

/**
 * @brief A wrapper around a {@link Message} which can be stored in a {@link MessageCache}. 
 *
 * This class adds Qz-specific members to a {@link Message}.
 */
public class MessageWrapper {

	/**
	 * Road class types.
	 */
	public static enum RoadClass {
		MOTORWAY,
		TRUNK,
		PRIMARY,
		SECONDARY,
		TERTIARY,
		OTHER,
		NONE,
	}

	/** Pattern for parsing data files */
	static Pattern colonPattern = Pattern.compile(";");

	/** The encapsulated message. */
	public final Message message;
	
	/** A unique identifier. If a message is replaced with a new one, the identifier is retained. */
	public final String id;

	/** IDs of other messages replaced by this message */
	public final List<String> replacedIds;

	/** When the message was first received. */
	public final Date received;

	/** Time zone for {@link #received} **/
	public final TimeZone receivedTz;

	/** TMC to TraFF event mapping */
	private static Map<Integer, List<TraffMapping>> MAPPINGS = new HashMap<Integer, List<TraffMapping>>();
	static {
		try {
			BufferedReader br = new BufferedReader(new InputStreamReader(MessageWrapper.class.getResourceAsStream("event.dat")));
			// first line contains column headers, skip
			br.readLine();
			String line;
			while ((line = br.readLine()) != null)
				try {
					TraffMapping mapping = new TraffMapping(line);
					if (!MAPPINGS.containsKey(mapping.tmcId))
						MAPPINGS.put(mapping.tmcId, new ArrayList<TraffMapping>());
					MAPPINGS.get(mapping.tmcId).add(mapping);
				} catch (IllegalArgumentException e) {
					// NOP (invalid lines are skipped)
				}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @brief Creates a new message wrapper using data from an existing message.
	 * 
	 * This constructor is intended for use with messages which replace an existing message. The new message wrapper is
	 * assigned the specified identifier, while the identifiers of all other messages in {@code messagesToRemove} are
	 * stored in the {@ref #replacedIds} member.
	 * 
	 * @param message The new message
	 * @param messagesToRemove Messages replaced by {@code message}
	 * @param id The identifier for the new message, inherited from one of the replaced messages
	 * @param received The timestamp when the first replaced message was received
	 * @param receivedTz The time zone for {@code received}
	 */
	public MessageWrapper(Message message, List<MessageWrapper> messagesToRemove, String id, Date received, TimeZone receivedTz) {
		this.message = message;

		List<String> ids = null;
		if (messagesToRemove != null)
			for (MessageWrapper wrapper : messagesToRemove)
				if (!wrapper.id.equals(id)) {
					if (ids == null)
						ids = new ArrayList<String>();
					ids.add(wrapper.id);
				}
		this.replacedIds = ids;

		this.id = id;
		this.received = received;
		this.receivedTz = receivedTz;
	}

	/**
	 * @brief Creates a new message wrapper for a new message.
	 * 
	 * This constructor is intended for use with messages that do not replace any existing messages, or replace
	 * multiple messages without a clear single “ancestor”.
	 * 
	 * The new message wrapper is assigned a new identifier that is guaranteed to be unique. In its simplest form, an
	 * identifier is of the form {@code tmc:a.1.0:a.1.32908.n.3,5} ({@code tmc} prefix, sending service ID, fully
	 * qualified location, direction and update classes). If that identifier is currently in use, it is extended with
	 * a dot and a number which is incremented until the resulting identifier is unique (e.g.
	 * {@code tmc:a.1.0:a.1.32908.n.3,5.2}).
	 * 
	 * Identifiers of the messages in {@code messagesToRemove} are stored in the {@ref #replacedIds} member.
	 * 
	 * @param message The new message
	 * @param messagesToRemove Messages replaced by {@code message}
	 * @param idsInUse Identifiers already used by currently active messages
	 */
	public MessageWrapper(Message message, List<MessageWrapper> messagesToRemove, Set<String> idsInUse) {
		this.message = message;
		this.received = message.getTimestamp();
		this.receivedTz = message.timeZone;
		
		/* build a fresh ID of the form tmc:a.1.0:a.1.32908.n.3,5.42 */
		Boolean isFirstUpdateClass = true;
		StringBuilder idBuilder = new StringBuilder("tmc:");
		idBuilder.append(String.format("%01x.%d.%d:", message.cc, message.getLocationTableNumber(), message.getSid()));
		if (message.interroad)
			idBuilder.append(String.format("%01x.%d.%d.", message.fcc, message.getForeignLocationTableNumber(), message.lcid));
		else
			idBuilder.append(String.format("%01x.%d.%d.", message.cc, message.getLocationTableNumber(), message.lcid));
		idBuilder.append(message.direction != 0 ? "n" : "p");
		long updateClasses = 0;
		for (int evtCode : message.getEvents())
			updateClasses |= 1L << (TMC.getEvent(evtCode).updateClass - 1);
		for (int i = 1; i <= 39; i++)
			if ((updateClasses & (1L << (i - 1))) != 0) {
				if (isFirstUpdateClass) {
					idBuilder.append(".").append(i);
					isFirstUpdateClass = false;
				} else
					idBuilder.append(",").append(i);
			}
		String tmpId = idBuilder.toString();
		int i = 0;
		while (idsInUse.contains(tmpId)) {
			i++;
			tmpId = String.format("%s.%d", idBuilder.toString(), i);
		}
		this.id = tmpId;

		List<String> ids = null;
		if (messagesToRemove != null)
			for (MessageWrapper wrapper : messagesToRemove)
				if (!wrapper.id.equals(id)) {
					if (ids == null)
						ids = new ArrayList<String>();
					ids.add(wrapper.id);
				}
		this.replacedIds = ids;
	}

	/**
	 * @brief Returns the road class for the location of the message.
	 * 
	 * TMC has no dedicated type for trunk roads: Some countries may classify motorway-grade trunk roads as motorways
	 * for TMC purposes. Other countries may classify them as order 1 roads ({@code RoadClass#PRIMARY}). In this case,
	 * the next-lower road level may either be classified as the same level, or as {@code RoadClass#SECONDARY}.
	 * 
	 * For this reason, {@code RoadClass#TRUNK} will never be returned for some countries.
	 * 
	 * Also, TMC knows only two types of ring roads: motorway and non-motorway. For ring roads of the latter type,
	 * {@code RoadClass#OTHER} will be returned.
	 * 
	 * @param message
	 * 
	 * @return The road class, or {@code RoadClass#NONE} if the location does not refer to a road
	 */
	public static RoadClass getRoadClass(Message message) {
		TMCLocation location = message.location;
		boolean hasTrunkRoads = MessageWrapper.hasTrunkRoads(message.fcc, message.getForeignLocationTableNumber());

		while (location != null) {
			if (LocationClass.LINE.equals(location.category)) {
				if (location.tcd == 1) {
					switch (location.stcd) {
					case 1:
						return RoadClass.MOTORWAY;
					case 2:
						return hasTrunkRoads ? RoadClass.TRUNK : RoadClass.PRIMARY;
					case 3:
						return hasTrunkRoads ? RoadClass.PRIMARY : RoadClass.SECONDARY;
					case 4:
						return hasTrunkRoads ? RoadClass.SECONDARY: RoadClass.TERTIARY;
					default:
						return RoadClass.OTHER;
					}
				} else if ((location.tcd == 2) && (location.stcd == 1)) {
					return RoadClass.MOTORWAY;
				}
			}

			if (location instanceof TMCPoint) {
				if (((TMCPoint) location).road != null)
					location = ((TMCPoint) location).road;
				else
					location = ((TMCPoint) location).segment;
			} else if (location instanceof Segment) {
				if (((Segment) location).road != null)
					location = ((Segment) location).road;
				else
					location = ((Segment) location).segment;
			} else
				return RoadClass.NONE;
		}

		return RoadClass.NONE;
	}

	/**
	 * @brief Whether the location table has a category for trunk roads.
	 * 
	 * @param cc The country code
	 * @param ltn The location table number
	 * 
	 * @return
	 */
	public static boolean hasTrunkRoads(int cc, int ltn) {
		// TODO complete this table
		switch(cc << 8 | ltn) {
		// TODO Belgium
		// TODO Denmark
		// TODO Netherlands
		// TODO Norway
		// TODO Sweden
		// TODO Slovakia
		// case 0x219: /* 2/25: Czech Republic */
		case 0x409: /* 4/9: Switzerland */
			return false;
		case 0x501: /* 5/1: Italy */
			return true;
			// case 0x611: /* 6/17: Finland */
		case 0xa01: /* A/1: Austria */
			return true;
			// case 0xc07: /* C/7: UK */
		case 0xd01: /* D/1: Germany */
			return false;
			// case 0xe11: /* E/17: Spain */
			// case 0xf20: /* F/32: France */
		}
		return true;
	}

	/**
	 * @brief Returns the road class for the location of the message.
	 * 
	 * See {@code #getRoadClass(Message)} for details.
	 * 
	 * @return The road class, or {@code RoadClass#NONE} if the location does not refer to a road
	 */
	public RoadClass getRoadClass() {
		return getRoadClass(message);
	}

	/**
	 * @brief Converts a message to TraFF format.
	 * 
	 * Since not all TMC features are supported in TraFF, some messages cannot be converted. These include:
	 * <ul>
	 * <li>Messages whose primary location is not a point location</li>
	 * <li>Messages that do not contain any events supported in TraFF</li>
	 * </ul> 
	 * 
	 * @return The message in TraFF XML format, or null if the message cannot be converted
	 */
	public String toXml() {
		if ((message.location == null) || !(message.location instanceof TMCPoint))
			return null;

		StringBuilder builder = new StringBuilder();
		DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXXX", Locale.ROOT);
		df.setTimeZone(receivedTz);

		builder.append(String.format("  <message id=\"%s\" receive_time=\"%s\"", id, df.format(received)));
		df.setTimeZone(TimeZone.getTimeZone(message.timeZone.getID()));
		builder.append(String.format(" update_time=\"%s\"", df.format(message.getTimestamp())));
		if (message.isCancellation()) {
			builder.append(" cancellation=\"true\"");
			builder.append(">\n");
		} else {
			if (message.startTime != -1)
				builder.append(String.format(" start_time=\"%s\"", df.format(message.startTime)));
			if (message.stopTime != -1)
				builder.append(String.format(" end_time=\"%s\"", df.format(message.stopTime)));
			builder.append(String.format(" expiration_time=\"%s\"", df.format(message.getPersistence())));
			/* FIXME make method visible
			if (message.isForecastMessage())
				builder.append(" forecast=\"true\"");
			 */
			if (message.urgency == EventUrgency.URGENT)
				builder.append(" urgency=\"URGENT\"");
			else if (message.urgency == EventUrgency.XURGENT)
				builder.append(" urgency=\"X_URGENT\"");
		}
		builder.append(">\n");

		if ((this.replacedIds != null) && !this.replacedIds.isEmpty()) {
			builder.append("    <merge>\n");
			for (String replacedId : replacedIds)
				builder.append(String.format("      <replaces id=\"%s\"/>\n", replacedId));
			builder.append("    </merge>\n");
		}

		if (message.isCancellation()) {
			builder.append("  </message>\n");
			return builder.toString();
		}

		/* location */
		builder.append("    <location fuzziness=\"LOW_RES\"");
		// TODO location.destination
		if (message.isBidirectional)
			builder.append(" directionality=\"BOTH_DIRECTIONS\"");
		else
			builder.append(" directionality=\"ONE_DIRECTION\"");
		// FIXME location.ramps
		switch (getRoadClass()) {
		case MOTORWAY:
			builder.append(" road_class=\"MOTORWAY\"");
			break;
		case TRUNK:
			builder.append(" road_class=\"TRUNK\"");
			break;
		case PRIMARY:
			builder.append(" road_class=\"PRIMARY\"");
			break;
		case SECONDARY:
			builder.append(" road_class=\"SECONDARY\"");
			break;
		case TERTIARY:
			builder.append(" road_class=\"TERTIARY\"");
			break;
		case OTHER:
			builder.append(" road_class=\"OTHER\"");
			break;
		default:
			/* NOP */
			break;
		}
		// TODO location.road_is_urban
		String roadRef = message.getRoadNumber();
		if (roadRef != null)
			builder.append(String.format(" road_ref=\"%s\"", roadRef));
		if ((message.location != null) && (message.location.roadName != null))
			builder.append(String.format(" road_name=\"%s\"", message.location.roadName.name));
		builder.append(">\n");

		TMCPoint from = null, to = null, at = null, via = null, not_via = null, aux = null;

		boolean isOnRingRoad = false;

		/* figure out if we’re on a ring road (L2.* location class) */
		if ((((TMCPoint) message.location).road != null)
				&& (((TMCPoint) message.location).road.category == LocationClass.LINE)
				&& (((TMCPoint) message.location).road.tcd == 2))
			isOnRingRoad = true;

		if (message.extent > 0) {
			/* nonzero extent, we have a distinct from and to point */
			from = (TMCPoint) message.getSecondaryLocation();
			to = (TMCPoint) message.location;
			/* if the location refers to a ring road, add via (half the extent) or not_via (if extent is 1) */
			if (isOnRingRoad) {
				if (message.extent > 1)
					via = to.getOffset(message.extent / 2, message.direction);
				else
					not_via = from.getOffset(1, message.direction);
			}
		} else {
			at = (TMCPoint) message.location;
			/* add from or to (or both for ring roads) */
			aux = at.getOffset(1, message.direction);
			if (!aux.equals(at))
				from = aux;
			if (aux.equals(at) || isOnRingRoad) {
				aux = at.getOffset(1, message.direction == 0 ? 1 : 0);
				if (!aux.equals(at))
					to = aux;
			}
		}

		appendLocation(builder, from, "from");
		appendLocation(builder, to, "to");
		appendLocation(builder, at, "at");
		appendLocation(builder, via, "via");
		appendLocation(builder, not_via, "not_via");

		builder.append("    </location>\n");

		/* events */
		boolean hasEvents = false;
		builder.append("    <events");
		builder.append(">\n");
		for (InformationBlock ib : message.getInformationBlocks())
			for (AlertC.Event ev : ib.getEvents())
				hasEvents |= appendEvent(builder, ib, ev);
		if (!hasEvents)
			return null;
		builder.append("    </events>\n");

		builder.append("  </message>\n");
		return builder.toString();
	}

	/**
	 * @brief Appends an event to the TraFF {@link StringBuilder} for a message.
	 * 
	 * If no TraFF mapping for the event exists, nothing is appended and false is returned.
	 * 
	 * @param builder The string builder
	 * @param ib The information block which contains {@code event}
	 * @param event The event to append
	 * @return True if data was appended, false if not
	 */
	private static boolean appendEvent(StringBuilder builder, InformationBlock ib, AlertC.Event event) {
		List<TraffMapping> mappings = MAPPINGS.get(event.tmcEvent.code);
		if ((mappings == null) || (mappings.isEmpty()))
			return false;
		for (TraffMapping mapping : mappings) {
			builder.append(String.format("      <event class=\"%s\" type=\"%s\"", mapping.traffClass, mapping.traffEvent));
			if (mapping.length >= 0)
				builder.append(String.format(" length=\"%d\"", mapping.length));
			if (mapping.speed >= 0)
				builder.append(String.format(" speed=\"%d\"", mapping.speed));
			// TODO quantifier (implicit and explicit)
			builder.append(">\n");
			if ((mapping.siClass != null) && (mapping.si != null))
				for (String siType : mapping.si) {
					builder.append(String.format("        <supplementary_info class=\"%s\" type=\"%s\"/>\n",
							mapping.siClass, siType));
				}
			// TODO supplementary information explicitly supplied (class, type, q:*)
			builder.append("      </event>\n");
		}
		return true;
	}

	/**
	 * @brief Appends a location to the TraFF {@link StringBuilder} for a message.
	 * 
	 * If the location is null, nothing is appended.
	 * 
	 * @param builder The string builder
	 * @param location The location to append
	 * @param tag The tag to use for the location
	 */
	private static void appendLocation(StringBuilder builder, TMCPoint location, String tag) {
		if (location != null) {
			builder.append("      <").append(tag);
			if ((location.name1 != null) && (location.name1.name != null))
				builder.append(" junction_name=\"").append(location.name1.name).append("\"");
			if (location.junctionNumber != null)
				builder.append(" junction_ref=\"").append(location.junctionNumber).append("\"");
			builder.append(">");
			builder.append(String.format("%+f %+f", location.yCoord, location.xCoord));
			builder.append("</").append(tag).append(">\n");
		}
	}

	/**
	 * A mapping which specifies how a TMC event maps to one or multiple TraFF events.
	 * 
	 * If a single TMC event is represented by multiple TraFF events, one {@code TraffMapping} instance exists for each
	 * TraFF event.
	 */
	private static class TraffMapping {
		static Pattern pipePattern = Pattern.compile("\\|");

		/** The TMC event code */
		public final int tmcId;

		/** The mnemonic for the TraFF event class */
		public final String traffClass;

		/** The TraFF event mnemonic */
		public final String traffEvent;

		/** The speed in km/h, -1 if not specified */
		public final int speed;

		/** The length of the route affected, in m, -1 if not specified */
		public final int length;

		/** Whether the event is a forecast */
		public final boolean isForecast;

		/** Supplementary information class, or null if none is specified */
		public final String siClass;

		/** Supplementary information, or null if none is specified */
		public final List<String> si;

		TraffMapping(String line) throws IllegalArgumentException {
			String[] comp = colonPattern.split(line);
			if (comp.length < 3)
				throw new IllegalArgumentException("too few data fields in line");
			this.tmcId = Integer.parseInt(comp[0]);
			this.traffClass = comp[1].trim();
			this.traffEvent = comp[2].trim();
			if ((comp.length < 4) || (comp[3].isEmpty()))
				this.speed = -1;
			else
				this.speed = Integer.parseInt(comp[3]);
			if ((comp.length < 5) || (comp[4].isEmpty()))
				this.length = -1;
			else
				this.length = Integer.parseInt(comp[4]);
			// TODO qType (5)
			// TODO quantifier (6)
			if ((comp.length < 8) || (comp[7].isEmpty()))
				this.isForecast = false;
			else
				this.isForecast = Boolean.parseBoolean(comp[7]);

			/*
			 * Note: while we do allow multiple SIs per TraFF event generated from a single TMC event, all these SIs
			 * must be of the same class. There is only one TMC event (1062) which results in a TraFF event with
			 * multiple SIs, and these happen to be all of the same class.
			 */
			if ((comp.length < 9) || (comp[8].isEmpty()))
				this.siClass = null;
			else
				this.siClass = comp[8];
			if ((comp.length < 10) || (comp[9].isEmpty()))
				this.si = null;
			else {
				String[] siArray = pipePattern.split(comp[9]);
				this.si = Arrays.asList(siArray);
			}
		}
	}
}
