"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chalk_1 = __importDefault(require("chalk"));
const cli_cursor_1 = __importDefault(require("cli-cursor"));
const elegant_spinner_1 = __importDefault(require("elegant-spinner"));
const figures_1 = __importDefault(require("figures"));
const indent_string_1 = __importDefault(require("indent-string"));
const log_update_1 = __importDefault(require("log-update"));
class MultiLineRenderer {
    constructor(tasks, options) {
        this.tasks = tasks;
        this.options = options;
        this.indentation = 2;
        this.bottomBar = {};
    }
    render() {
        if (this.id) {
            return;
        }
        cli_cursor_1.default.hide();
        this.id = setInterval(() => {
            log_update_1.default(this.multiLineRenderer(this.tasks), this.renderBottomBar(), this.renderPrompt());
        }, 100);
    }
    end() {
        if (this.id) {
            clearInterval(this.id);
            this.id = undefined;
        }
        log_update_1.default(this.multiLineRenderer(this.tasks), this.renderBottomBar());
        if (this.options.clearOutput) {
            log_update_1.default.clear();
        }
        else {
            log_update_1.default.done();
        }
        cli_cursor_1.default.show();
    }
    multiLineRenderer(tasks, level = 0) {
        let output = [];
        for (const task of tasks) {
            if (task.isEnabled()) {
                if (task.hasTitle()) {
                    if (task.isSkipped() && (task === null || task === void 0 ? void 0 : task.options.collapseSkips)) {
                        task.title = !task.isSkipped() ? `${task === null || task === void 0 ? void 0 : task.title}` : `${task === null || task === void 0 ? void 0 : task.output} ${chalk_1.default.dim('[SKIPPED]')}`;
                    }
                    if (!(tasks.some((task) => task.hasFailed()) && !task.hasFailed() && task.options.exitOnError !== false && !(task.isCompleted() || task.isSkipped()))) {
                        output.push(this.formatString(task.title, this.getSymbol(task), level));
                    }
                    else {
                        output.push(this.formatString(task.title, chalk_1.default.red(figures_1.default.squareSmallFilled), level));
                    }
                }
                if (task === null || task === void 0 ? void 0 : task.output) {
                    if (task.isPending() && task.isPrompt()) {
                        this.promptBar = task.output;
                    }
                    else if (task.isBottomBar() || !task.hasTitle()) {
                        const data = this.dumpData(task, -1);
                        if (!this.bottomBar[task.id]) {
                            this.bottomBar[task.id] = {};
                            this.bottomBar[task.id].data = [];
                            this.bottomBar[task.id].items = typeof task.options.bottomBar === 'boolean' ? 1 : task.options.bottomBar;
                        }
                        if (!(data === null || data === void 0 ? void 0 : data.some((element) => this.bottomBar[task.id].data.includes(element)))) {
                            this.bottomBar[task.id].data = [...this.bottomBar[task.id].data, ...data];
                        }
                    }
                    else if (task.isPending() || task.haspersistentOutput()) {
                        output = [...output, ...this.dumpData(task, level)];
                    }
                    else if (task.isSkipped() && task.options.collapseSkips === false) {
                        output = [...output, ...this.dumpData(task, level)];
                    }
                }
                if ((task.isPending() || task.hasFailed()
                    || task.isCompleted() && !task.hasTitle()
                    || task.isCompleted() && task.options.collapse === false && task.hasSubtasks() && !task.subtasks.some((subtask) => subtask.options.collapse === true)
                    || task.isCompleted() && task.hasSubtasks() && task.subtasks.some((subtask) => subtask.options.collapse === false)
                    || task.isCompleted() && task.hasSubtasks() && task.subtasks.some((subtask) => subtask.hasFailed()))
                    && task.options.showSubtasks !== false && task.hasSubtasks()) {
                    const subtaskLevel = !task.hasTitle() ? level : level + 1;
                    const subtaskRender = this.multiLineRenderer(task.subtasks, subtaskLevel);
                    if (subtaskRender !== '') {
                        output = [...output, subtaskRender];
                    }
                }
                if (task.isCompleted() || task.hasFailed()) {
                    this.promptBar = null;
                    if (task.hasFailed() || (!task.hasTitle() || task.isBottomBar()) && task.haspersistentOutput() !== true) {
                        delete this.bottomBar[task.id];
                    }
                }
            }
        }
        return output.join('\n');
    }
    renderBottomBar() {
        if (Object.keys(this.bottomBar).length > 0) {
            this.bottomBar = Object.keys(this.bottomBar).reduce((o, key) => {
                if (!(o === null || o === void 0 ? void 0 : o[key])) {
                    o[key] = {};
                }
                o[key].data = this.bottomBar[key].data.slice(-this.bottomBar[key].items);
                return o;
            }, {});
            const returnRender = Object.values(this.bottomBar).reduce((o, value) => o = [...o, ...value.data], []);
            return ['\n', ...returnRender].join('\n');
        }
    }
    renderPrompt() {
        if (this.promptBar) {
            return `\n\n${this.promptBar}`;
        }
    }
    dumpData(task, level) {
        const output = [];
        if (typeof task.output === 'string') {
            task.output.split('\n').filter(Boolean).forEach((line, i) => {
                const icon = i === 0 ? this.getSymbol(task, true) : ' ';
                output.push(this.formatString(line, icon, level + 1));
            });
        }
        return output;
    }
    formatString(string, icon, level) {
        return `${indent_string_1.default(`${icon} ${string}`, level * this.indentation)}`;
    }
    getSymbol(task, data = false) {
        if (!task.spinner && !data) {
            task.spinner = elegant_spinner_1.default();
        }
        if (task.isPending() && !data) {
            return task.options.showSubtasks !== false && task.hasSubtasks() ? chalk_1.default.yellow(figures_1.default.pointer) : chalk_1.default.yellowBright(task.spinner());
        }
        if (task.isCompleted() && !data) {
            if (task.hasSubtasks() && task.subtasks.some((subtask) => subtask.hasFailed())) {
                return chalk_1.default.yellow(figures_1.default.warning);
            }
            return chalk_1.default.green(figures_1.default.tick);
        }
        if (task.hasFailed() && !data) {
            return task.hasSubtasks() ? chalk_1.default.red(figures_1.default.pointer) : chalk_1.default.red(figures_1.default.cross);
        }
        if (task.isSkipped() && !data && task.options.collapseSkips === false) {
            return chalk_1.default.yellow(figures_1.default.warning);
        }
        else if (task.isSkipped() && (data || task.options.collapseSkips)) {
            return chalk_1.default.yellow(figures_1.default.arrowDown);
        }
        if (task.isPrompt()) {
            return chalk_1.default.cyan(figures_1.default.questionMarkPrefix);
        }
        if (!data) {
            return chalk_1.default.dim(figures_1.default.squareSmallFilled);
        }
        else {
            return figures_1.default.pointerSmall;
        }
    }
}
exports.MultiLineRenderer = MultiLineRenderer;
MultiLineRenderer.nonTTY = false;
//# sourceMappingURL=default.renderer.js.map