package com.hardcodecoder.pulsemusic.tag;

import androidx.annotation.NonNull;

import com.hardcodecoder.pulsemusic.model.AudioFileModel;
import com.hardcodecoder.pulsemusic.model.TagModel;
import com.hardcodecoder.pulsemusic.utils.LogUtils;

import org.jaudiotagger.audio.AudioFile;
import org.jaudiotagger.audio.AudioFileIO;
import org.jaudiotagger.audio.exceptions.CannotReadException;
import org.jaudiotagger.audio.exceptions.CannotWriteException;
import org.jaudiotagger.audio.exceptions.InvalidAudioFrameException;
import org.jaudiotagger.audio.exceptions.ReadOnlyFileException;
import org.jaudiotagger.tag.FieldKey;
import org.jaudiotagger.tag.Tag;
import org.jaudiotagger.tag.TagException;
import org.jaudiotagger.tag.images.AndroidArtwork;
import org.jaudiotagger.tag.images.Artwork;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.Callable;

public class AudioTagWriter implements Callable<Boolean> {

    private static final String TAG = AudioTagWriter.class.getSimpleName();
    private final AudioFileModel mAudioFileModel;

    public AudioTagWriter(@NonNull AudioFileModel audioFileModel) {
        mAudioFileModel = audioFileModel;
    }

    @Override
    public Boolean call() {
        return writeTag();
    }

    private boolean writeTag() {
        try {
            File file = new File(mAudioFileModel.getAbsolutePath());
            AudioFile audioFile = AudioFileIO.read(file);
            Tag tag = audioFile.getTag();
            TagModel tagModel = mAudioFileModel.getTagModel();
            tag.setField(FieldKey.TITLE, tagModel.getTitle());
            tag.setField(FieldKey.ALBUM, tagModel.getAlbum());
            tag.setField(FieldKey.ALBUM_ARTIST, tagModel.getAlbumArtist());
            tag.setField(FieldKey.ARTIST, tagModel.getArtist());
            tag.setField(FieldKey.TRACK, String.valueOf(tagModel.getTrackNumber()));
            tag.setField(FieldKey.TRACK_TOTAL, String.valueOf(tagModel.getTrackCount()));
            tag.setField(FieldKey.DISC_NO, String.valueOf(tagModel.getDiscNumber()));
            tag.setField(FieldKey.DISC_TOTAL, String.valueOf(tagModel.getDiscCount()));
            tag.setField(FieldKey.YEAR, tagModel.getYear());
            tag.setField(FieldKey.GENRE, tagModel.getGenre());
            tag.setField(FieldKey.LYRICS, tagModel.getLyrics());

            String workingDirPath = new File(mAudioFileModel.getAbsolutePath()).getParent();
            if (null != workingDirPath) {
                File artworkFile = TagUtils.getArtworkFile(workingDirPath);
                if (artworkFile.exists()) {
                    tag.deleteArtworkField();
                    Artwork artwork = AndroidArtwork.createArtworkFromFile(artworkFile);
                    tag.setField(artwork);
                }
            }
            audioFile.setTag(tag);
            AudioFileIO.write(audioFile);
            return true;
        } catch (CannotReadException | InvalidAudioFrameException | TagException | IOException |
                ReadOnlyFileException | CannotWriteException e) {
            LogUtils.logException(LogUtils.Type.IO, TAG, "at: call()", e);
            return false;
        }
    }
}