package com.hardcodecoder.pulsemusic.tag;

import android.content.ContentUris;
import android.content.Context;
import android.database.Cursor;
import android.media.MediaMetadataRetriever;
import android.net.Uri;
import android.provider.MediaStore.Audio.Media;
import android.provider.OpenableColumns;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hardcodecoder.pulsemusic.R;
import com.hardcodecoder.pulsemusic.model.AudioFileModel;
import com.hardcodecoder.pulsemusic.model.TagModel;
import com.hardcodecoder.pulsemusic.utils.LogUtils;
import com.hardcodecoder.pulsemusic.utils.PulseIOUtils;

import org.jaudiotagger.audio.AudioFile;
import org.jaudiotagger.audio.AudioFileIO;
import org.jaudiotagger.audio.exceptions.CannotReadException;
import org.jaudiotagger.audio.exceptions.InvalidAudioFrameException;
import org.jaudiotagger.audio.exceptions.ReadOnlyFileException;
import org.jaudiotagger.tag.FieldKey;
import org.jaudiotagger.tag.Tag;
import org.jaudiotagger.tag.TagException;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.concurrent.Callable;

import static com.hardcodecoder.pulsemusic.utils.LogUtils.Type.IO;

public class AudioTagReader implements Callable<AudioFileModel> {

    private static final String TAG = AudioTagReader.class.getSimpleName();
    private final WeakReference<Context> mContext;
    private final Uri mUri;
    private final int mId;

    public AudioTagReader(@NonNull Context context, int trackId) {
        mContext = new WeakReference<>(context);
        mId = trackId;
        mUri = ContentUris.withAppendedId(Media.EXTERNAL_CONTENT_URI, trackId);
    }

    @Override
    @Nullable
    public AudioFileModel call() {
        Context context = mContext.get();
        if (null == context) return null;

        String displayName = getDisplayName();
        if (null == displayName) return null;

        File file = getAudioFile(context, displayName);
        if (null == file || !file.exists()) return null;

        String absolutePath = file.getAbsolutePath();

        TagModel tag = tryReadingFromJAudioTagger(file);
        if (null == tag) tag = tryReadingFromMediaMetadataRetriever(context);

        String mineType = context.getContentResolver().getType(mUri);
        return new AudioFileModel(mId, displayName, absolutePath, mineType, tag);
    }

    @Nullable
    private String getDisplayName() {
        Context context = mContext.get();
        Cursor cursor = context.getContentResolver()
                .query(mUri,
                        null,
                        null,
                        null,
                        null);
        int nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
        String displayName = null;
        if (cursor.moveToFirst()) displayName = cursor.getString(nameIndex);
        cursor.close();
        return displayName;
    }

    @Nullable
    private File getAudioFile(@NonNull Context context, @NonNull String displayName) {
        String workingDirPath = TagUtils.getOrCreateWorkingDirectoryPath(context);
        if (null == workingDirPath || workingDirPath.isEmpty()) return null;

        File outputFile = null;
        BufferedOutputStream bufferedOutputStream = null;

        try (BufferedInputStream bufferedInputStream = new BufferedInputStream(
                context.getContentResolver().openInputStream(mUri))) {

            outputFile = new File(workingDirPath + File.separator + displayName);
            bufferedOutputStream = new BufferedOutputStream(new FileOutputStream(outputFile));

            int b;
            while ((b = bufferedInputStream.read()) != -1)
                bufferedOutputStream.write(b);
        } catch (Exception e) {
            LogUtils.logException(IO, TAG, "at: getAudioFile()", e);
        } finally {
            PulseIOUtils.closeBufferedOutputStream(bufferedOutputStream);
        }
        return outputFile;
    }

    @Nullable
    private TagModel tryReadingFromJAudioTagger(@NonNull File file) {
        try {
            AudioFile f = AudioFileIO.read(file);
            Tag tag = f.getTagOrCreateDefault();

            String title = tag.getFirst(FieldKey.TITLE);
            String album = tag.getFirst(FieldKey.ALBUM);
            String albumArtist = tag.getFirst(FieldKey.ALBUM_ARTIST);
            String artist = tag.getFirst(FieldKey.ARTIST);
            int trackNumber = TagUtils.parseIntOrDefault(tag.getFirst(FieldKey.TRACK));
            int trackCount = TagUtils.parseIntOrDefault(tag.getFirst(FieldKey.TRACK_TOTAL));
            int discNumber = TagUtils.parseIntOrDefault(tag.getFirst(FieldKey.DISC_NO));
            int discCount = TagUtils.parseIntOrDefault(tag.getFirst(FieldKey.DISC_TOTAL));
            String year = tag.getFirst(FieldKey.YEAR);
            String genre = tag.getFirst(FieldKey.GENRE);
            String lyrics = tag.getFirst(FieldKey.LYRICS);

            return new TagModel(
                    title,
                    album,
                    albumArtist,
                    artist,
                    trackNumber,
                    trackCount,
                    discNumber,
                    discCount,
                    year,
                    genre,
                    lyrics
            );

        } catch (CannotReadException | IOException | ReadOnlyFileException | InvalidAudioFrameException | TagException e) {
            LogUtils.logException(IO, TAG, "at: readTag(): " + file.getAbsolutePath(), e);
            return null;
        }
    }

    @NonNull
    private TagModel tryReadingFromMediaMetadataRetriever(@NonNull Context context) {
        MediaMetadataRetriever mmr = new MediaMetadataRetriever();
        mmr.setDataSource(context, mUri);
        String defText = context.getString(R.string.unknown);
        String title = defText;
        String album = defText;
        String albumArtist = defText;
        String artist = defText;
        int trackNumber = 0;
        int trackCount = 0;
        int discNumber = 0;
        int discCount = 0;
        String year = "";
        String genre = defText;
        String lyrics = "";
        try {
            title = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_TITLE);
            album = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_ALBUM);
            albumArtist = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_ALBUMARTIST);
            artist = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_ARTIST);

            String trackStr = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_CD_TRACK_NUMBER);
            int[] track = TagUtils.parseAudioTrackRepresentation(trackStr);
            trackNumber = track[0];
            trackCount = track[1];

            String discStr = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_DISC_NUMBER);
            int[] disc = TagUtils.parseAudioTrackRepresentation(discStr);
            discNumber = disc[0];
            discCount = disc[1];

            year = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_YEAR);
            genre = mmr.extractMetadata(MediaMetadataRetriever.METADATA_KEY_GENRE);
            mmr.release();
        } catch (Exception e) {
            e.printStackTrace();
        }

        return new TagModel(
                title,
                album,
                albumArtist,
                artist,
                trackNumber,
                trackCount,
                discNumber,
                discCount,
                year,
                genre,
                lyrics
        );
    }
}