package com.hardcodecoder.pulsemusic.tag;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore.Audio.Media;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hardcodecoder.pulsemusic.model.AudioFileModel;
import com.hardcodecoder.pulsemusic.model.TagModel;
import com.hardcodecoder.pulsemusic.utils.LogUtils;
import com.hardcodecoder.pulsemusic.utils.PulseIOUtils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.concurrent.Callable;

public class AudioFileUpdateTask implements Callable<Boolean> {

    private static final String TAG = AudioFileUpdateTask.class.getSimpleName();
    private final WeakReference<Context> mContextReference;
    private final AudioFileModel mAudioFileModel;

    public AudioFileUpdateTask(@NonNull Context context, @NonNull AudioFileModel audioFileModel) {
        mContextReference = new WeakReference<>(context);
        mAudioFileModel = audioFileModel;
    }

    @Override
    @Nullable
    public Boolean call() {
        return updateOriginal();
    }

    private boolean updateOriginal() {
        Context context = mContextReference.get();
        Uri uri = mAudioFileModel.getAudioUri();

        BufferedInputStream bufferedInputStream = null;
        boolean success = false;
        final ContentResolver resolver = context.getContentResolver();
        try (BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(resolver.openOutputStream(uri, "w"))) {
            setPending(resolver, uri, true);

            File audioFile = new File(mAudioFileModel.getAbsolutePath());
            bufferedInputStream = new BufferedInputStream(new FileInputStream(audioFile));

            int b;
            while ((b = bufferedInputStream.read()) != -1)
                bufferedOutputStream.write(b);

            updateMediaStore(resolver);

            setPending(resolver, uri, false);
            success = true;
        } catch (IOException e) {
            LogUtils.logException(LogUtils.Type.IO, TAG, "at: updateOriginal()", e);
        } finally {
            PulseIOUtils.closeBufferedInputStream(bufferedInputStream);
        }
        return success;
    }

    private void setPending(@NonNull ContentResolver resolver, @NonNull Uri uri, boolean pending) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            ContentValues values = new ContentValues();
            values.put(Media.IS_PENDING, pending ? 1 : 0);
            int updated = resolver.update(uri, values, null, null);
            if (updated <= 0) {
                LogUtils.logInfo(TAG, "at: setPending(), unable to set pending = " + pending);
            }
        }
    }

    private void updateMediaStore(@NonNull ContentResolver resolver) {
        ContentValues values = new ContentValues();
        TagModel tag = mAudioFileModel.getTagModel();
        values.put(Media.TITLE, tag.getTitle());
        values.put(Media.ALBUM, tag.getTitle());
        values.put(Media.ALBUM_ARTIST, tag.getTitle());
        values.put(Media.ARTIST, tag.getTitle());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            values.put(Media.DISC_NUMBER, tag.getDiscNumber());
            values.put(Media.CD_TRACK_NUMBER, tag.getTrackNumber());
            values.put(Media.NUM_TRACKS, tag.getTrackCount());
        } else {
            values.put(Media.TRACK, getEncodedTrackNumber(tag.getDiscNumber(), tag.getTrackNumber()));
        }
        values.put(Media.YEAR, tag.getYear());
        values.put(Media.GENRE, tag.getGenre());

        int updated = resolver.update(mAudioFileModel.getAudioUri(), values, null, null);
        if (updated < 1) {
            Log.e(TAG, "Nothing updated");
        }
    }

    private int getEncodedTrackNumber(int discNum, int trackNum) {
        try {
            return (1000 * discNum) + trackNum;
        } catch (Exception e) {
            LogUtils.logException(
                    LogUtils.Type.BACKGROUND, TAG,
                    "at: getEncodedTrackNumber(" + discNum + ", " + trackNum + ")",
                    e);
        }
        return 0;
    }
}