package com.hardcodecoder.pulsemusic.loaders;

import android.annotation.SuppressLint;
import android.content.ContentUris;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hardcodecoder.pulsemusic.model.MusicModel;

public class MediaStoreHelper {

    public static final Uri sArtworkUri = Uri.parse("content://media/external/audio/albumart");


    // MediaStore.Audio.Media.DURATION existed well before APi 29
    // Suppress lint
    @SuppressLint("InlinedApi")
    @Nullable
    public static String getSortOrderFor(@Nullable SortOrder sortOrder) {
        if (null == sortOrder) return null;
        switch (sortOrder) {
            case TITLE_ASC:
                return MediaStore.Audio.Media.TITLE + " COLLATE NOCASE ASC";
            case TITLE_DESC:
                return MediaStore.Audio.Media.TITLE + " COLLATE NOCASE DESC";
            case DURATION_ASC:
                return MediaStore.Audio.Media.DURATION + " ASC";
            case DURATION_DESC:
                return MediaStore.Audio.Media.DURATION + " DESC";
            case DATE_MODIFIED_ASC:
                return MediaStore.Audio.Media.DATE_MODIFIED + " ASC";
            case DATE_MODIFIED_DESC:
                return MediaStore.Audio.Media.DATE_MODIFIED + " DESC";
            case TRACK_NUMBER_ASC:
                return MediaStore.Audio.Media.TRACK + " ASC";
            case TRACK_NUMBER_DESC:
                return MediaStore.Audio.Media.TRACK + " DESC";
            default:
                return null;
        }
    }

    @Nullable
    public static String getSortOrderFor(@Nullable SortOrder.ALBUMS sortOrder) {
        if (sortOrder == null) return null;
        switch (sortOrder) {
            case TITLE_ASC:
                return MediaStore.Audio.Albums.ALBUM + " COLLATE NOCASE ASC";
            case TITLE_DESC:
                return MediaStore.Audio.Albums.ALBUM + " COLLATE NOCASE DESC";
            case ARTIST_ASC:
                return MediaStore.Audio.Albums.ARTIST + " COLLATE NOCASE ASC";
            case ARTIST_DESC:
                return MediaStore.Audio.Albums.ARTIST + " COLLATE NOCASE DESC";
            case ALBUM_DATE_FIRST_YEAR_ASC:
                return MediaStore.Audio.Albums.FIRST_YEAR + " ASC, " + MediaStore.Audio.Albums.ALBUM + " COLLATE NOCASE ASC";
            case ALBUM_DATE_FIRST_YEAR_DESC:
                return MediaStore.Audio.Albums.FIRST_YEAR + " DESC, " + MediaStore.Audio.Albums.ALBUM + " COLLATE NOCASE ASC";
            case ALBUM_DATE_LAST_YEAR_ASC:
                return MediaStore.Audio.Albums.LAST_YEAR + " ASC";
            case ALBUM_DATE_LAST_YEAR_DESC:
                return MediaStore.Audio.Albums.LAST_YEAR + " DESC";
            default:
                return null;
        }
    }

    @Nullable
    public static String getSortOrderFor(@Nullable SortOrder.ARTIST sortOrder) {
        if (sortOrder == null) return null;
        switch (sortOrder) {
            case TITLE_ASC:
                return MediaStore.Audio.Artists.ARTIST + " COLLATE NOCASE ASC";
            case TITLE_DESC:
                return MediaStore.Audio.Artists.ARTIST + " COLLATE NOCASE DESC";
            case NUM_OF_TRACKS_ASC:
                return MediaStore.Audio.Artists.NUMBER_OF_TRACKS + " ASC";
            case NUM_OF_TRACKS_DESC:
                return MediaStore.Audio.Artists.NUMBER_OF_TRACKS + " DESC";
            default:
                return null;
        }
    }

    @SuppressLint("InlinedApi")
    @NonNull
    public static String[] getMediaAudioColumns(int skdInt) {
        if (skdInt >= Build.VERSION_CODES.R) {
            return new String[]{
                    MediaStore.Audio.Media._ID,             // 0
                    MediaStore.Audio.Media.TITLE,           // 1
                    MediaStore.Audio.Media.ALBUM,           // 2
                    MediaStore.Audio.Media.ALBUM_ID,        // 3
                    MediaStore.Audio.Media.ARTIST,          // 4
                    MediaStore.Audio.Media.DISC_NUMBER,     // 5
                    MediaStore.Audio.Media.CD_TRACK_NUMBER, // 6
                    MediaStore.Audio.Media.DATE_ADDED,      // 7
                    MediaStore.Audio.Media.DATE_MODIFIED,   // 8
                    MediaStore.Audio.Media.DURATION,        // 9
            };
        } else {
            return new String[]{
                    MediaStore.Audio.Media._ID,             // 0
                    MediaStore.Audio.Media.TITLE,           // 1
                    MediaStore.Audio.Media.ALBUM,           // 2
                    MediaStore.Audio.Media.ALBUM_ID,        // 3
                    MediaStore.Audio.Media.ARTIST,          // 4
                    MediaStore.Audio.Media.TRACK,           // 5
                    MediaStore.Audio.Media.DATE_ADDED,      // 6
                    MediaStore.Audio.Media.DATE_MODIFIED,   // 7
                    MediaStore.Audio.Media.DURATION,        // 8
            };
        }
    }

    @NonNull
    public static MusicModel getModelFromCursor(@NonNull Cursor cursor, int id, int skdInt) {
        String trackTitle;
        String album;
        String artist;
        long albumId;
        long dateAdded;
        long dateModified;
        int[] discTrackNumber;
        int duration;

        if (skdInt >= Build.VERSION_CODES.R) {
            trackTitle = cursor.getString(1);
            album = cursor.getString(2);
            albumId = cursor.getLong(3);
            artist = cursor.getString(4);
            discTrackNumber = new int[]{
                    cursor.getInt(5),
                    cursor.getInt(6)};
            dateAdded = cursor.getLong(7);
            dateModified = cursor.getLong(8);
            duration = cursor.getInt(9);
        } else {
            trackTitle = cursor.getString(1);
            album = cursor.getString(2);
            albumId = cursor.getLong(3);
            artist = cursor.getString(4);
            int track = cursor.getInt(5);
            discTrackNumber = new int[]{track / 1000, track % 1000};
            dateAdded = cursor.getLong(6);
            dateModified = cursor.getLong(7);
            duration = cursor.getInt(8);
        }
        String trackUri = ContentUris.withAppendedId(MediaStore.Audio.Media.EXTERNAL_CONTENT_URI, id).toString();
        String albumArtUri = ContentUris.withAppendedId(sArtworkUri, albumId).toString();

        return new MusicModel(
                id,
                trackTitle,
                album == null ? "" : album,
                albumId,
                artist == null ? "" : artist,
                trackUri,
                albumArtUri,
                dateAdded,
                dateModified,
                discTrackNumber[0],
                discTrackNumber[1],
                duration);
    }

    @Nullable
    public static String getAbsoluteAudioFilePath(@NonNull Context context, @NonNull Uri uri) {
        Cursor cursor = context.getContentResolver().query(uri,
                new String[]{MediaStore.Audio.Media.DATA},
                null,
                null,
                null);
        if (null != cursor && cursor.moveToFirst()) {
            String path = cursor.getString(0);
            cursor.close();
            return path;
        }
        return null;
    }
}