package com.hardcodecoder.pulsemusic.loaders;

import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;

import com.hardcodecoder.pulsemusic.providers.ProviderManager;
import com.hardcodecoder.pulsemusic.utils.AppSettings;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

@RequiresApi(Build.VERSION_CODES.Q)
public class MasterListLoaderQ extends BaseMasterListLoader {

    private Set<Integer> mExcludeIds = null;

    public MasterListLoaderQ(@NonNull Context context,
                             @Nullable SortOrder sortOrder,
                             @Nullable String selection,
                             @Nullable String[] selectionArgs) {
        super(context, sortOrder, selection, selectionArgs);
    }

    @Override
    public void onReady(@NonNull Context context) {
        mExcludeIds = getMediaIdsToExclude(context);
    }

    @Override
    public boolean excludeItem(int id) {
        return mExcludeIds != null && mExcludeIds.contains(id);
    }

    @Nullable
    @Override
    public String getSelection(@NonNull Context context, @Nullable String baseSelection) {
        // Append duration filter
        return (baseSelection == null || baseSelection.trim().equals("") ? "" : baseSelection + " AND ")
                + MediaStore.Audio.Media.DURATION + " >= ?";
    }

    @Nullable
    @Override
    public String[] getSelectionArgs(@NonNull Context context, @Nullable String[] baseSelectionArgs) {
        final int completeArgsLength = (baseSelectionArgs == null ? 0 : baseSelectionArgs.length) + 1;
        String[] completeSelectionArgs = new String[completeArgsLength];

        if (baseSelectionArgs != null) {
            // Copy the existing selection args
            System.arraycopy(baseSelectionArgs, 0, completeSelectionArgs, 0, baseSelectionArgs.length);
        }

        // Copy args for duration filter
        int durationFilter = AppSettings.getFilterDuration(context);
        completeSelectionArgs[completeArgsLength - 1] = String.valueOf(durationFilter * 1000 /*Must be in mills*/);
        return completeSelectionArgs;
    }

    @Nullable
    private Set<Integer> getMediaIdsToExclude(@NonNull Context context) {
        // Append ignored folders selection
        List<String> ignoredFoldersList = ProviderManager.getIgnoredListProvider().getIgnoredList();
        int ignoredListSize = (ignoredFoldersList == null) ? 0 : ignoredFoldersList.size();

        String excludeSelection = getExcludeSelection(ignoredListSize);
        String[] excludeSelectionArgs = null;
        if (ignoredFoldersList != null && ignoredListSize > 0)
            excludeSelectionArgs = getExcludeSelectionArgs(ignoredFoldersList);

        final Uri uri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI;
        final String[] cursor_cols = {MediaStore.Audio.Media._ID};
        final Cursor cursor = context.getContentResolver().query(
                uri,
                cursor_cols,
                excludeSelection,
                excludeSelectionArgs,
                null);

        Set<Integer> excludeTracksId = null;

        if (cursor != null && cursor.moveToFirst()) {
            excludeTracksId = new HashSet<>();
            do {
                excludeTracksId.add(cursor.getInt(0));
            } while (cursor.moveToNext());
            cursor.close();
        }
        return excludeTracksId;
    }

    @NonNull
    private String getExcludeSelection(int ignoredListSize) {
        // Prepare selection
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder
                .append("((")
                .append(MediaStore.Audio.Media.VOLUME_NAME).append(" LIKE ?")
                .append(" AND ")
                .append(MediaStore.Audio.Media.RELATIVE_PATH).append(" LIKE ?")
                .append(")");
        for (int i = 0; i < ignoredListSize - 1; i++) {
            stringBuilder
                    .append(" OR ")
                    .append("(")
                    .append(MediaStore.Audio.Media.VOLUME_NAME).append(" LIKE ?")
                    .append(" AND ")
                    .append(MediaStore.Audio.Media.RELATIVE_PATH).append(" LIKE ?")
                    .append(")");
        }
        return stringBuilder.append(")").toString();
    }

    @NonNull
    private String[] getExcludeSelectionArgs(@NonNull List<String> ignoredFolders) {
        String[] selectionArgs = new String[2 * ignoredFolders.size()];
        int section = 0;
        for (int i = 0; i < ignoredFolders.size(); i++) {
            String path = ignoredFolders.get(i);
            // complete path is in the form primary_external:Dir/Dir/Dir/
            // We split the volume name and relative path
            String[] segments = path.split(":");
            // Volume name
            selectionArgs[section++] = segments[0];
            // We ignore every sub directory inside
            // the directory this relative path points to
            selectionArgs[section++] = "%".concat(segments[1]).concat("%");
        }
        return selectionArgs;
    }
}