package com.hardcodecoder.pulsemusic.helper;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.net.Uri;

import androidx.annotation.ColorInt;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hardcodecoder.pulsemusic.MediaArtCache;
import com.hardcodecoder.pulsemusic.R;
import com.hardcodecoder.pulsemusic.model.MusicModel;
import com.hardcodecoder.pulsemusic.themes.ThemeColors;
import com.hardcodecoder.pulsemusic.utils.AppSettings;
import com.hardcodecoder.pulsemusic.utils.ImageUtil;

import java.io.InputStream;

public class MediaArtHelper {

    private static TypedArray mMediaArtColors;

    @Nullable
    public static Bitmap getAlbumArt(@NonNull Context context, @NonNull MusicModel data) {
        // We know that manually selected tracks have negative track id's
        if (data.getId() < 0)
            return getDefaultAlbumArtBitmap(context, data.getAlbumId());
        try {
            Uri uri;
            InputStream stream;
            if (AppSettings.ignoreMediaStoreArtwork(context)) {
                uri = Uri.parse(data.getTrackPath());
                stream = ImageUtil.getEmbeddedArtwork(context, uri);
            } else {
                uri = Uri.parse(data.getAlbumArtUrl());
                stream = context.getContentResolver().openInputStream(uri);
            }
            // Users complaint about poor quality artwork in lock screen
            // We used to load 512 x 512 artworks. is it too low quality?
            Bitmap bm = ImageUtil.getScaledBitmap(stream, 1024, 1024);
            return bm == null ? getDefaultAlbumArtBitmap(context, data.getAlbumId()) : bm;
        } catch (Exception e) {
            return getDefaultAlbumArtBitmap(context, data.getAlbumId());
        }
    }

    @Nullable
    public static Drawable getDefaultAlbumArt(@NonNull Context context, long albumId) {
        // albumId = -1 represents album art tinted with the primary color
        // So, we need not convert it into positive int
        if (albumId != -1) albumId = Math.abs(albumId);

        if (null != MediaArtCache.getMediaArtDrawable(albumId))
            return MediaArtCache.getMediaArtDrawable(albumId);
        Drawable drawable = ImageUtil.generateTintedDefaultAlbumArt(context, getTintColor(context, albumId));
        if (null == drawable) return null;
        MediaArtCache.addDrawableIfNotPresent(drawable, albumId);
        return drawable;
    }

    @Nullable
    public static Bitmap getDefaultAlbumArtBitmap(@NonNull Context context, long albumId) {
        // -1 represents the primary color tinted album art
        // So, we need not convert it into positive int
        if (albumId != -1) albumId = Math.abs(albumId);

        Drawable drawable = ImageUtil.generateTintedDefaultAlbumArt(context, getTintColor(context, albumId));
        if (null == drawable) return null;
        return ImageUtil.getBitmapFromDrawable(drawable);
    }

    @ColorInt
    private static int getTintColor(@NonNull Context context, long albumId) {
        if (albumId == -1) return ThemeColors.getCurrentColorPrimary();
        if (null == mMediaArtColors)
            mMediaArtColors = context.getResources().obtainTypedArray(R.array.album_art_colors);
        return mMediaArtColors.getColor((int) (albumId % mMediaArtColors.length()), 0xFF0000);
    }
}