package com.hardcodecoder.pulsemusic.activities.main;

import android.app.ProgressDialog;
import android.app.RecoverableSecurityException;
import android.content.Intent;
import android.content.IntentSender;
import android.graphics.drawable.ColorDrawable;
import android.media.MediaScannerConnection;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.provider.MediaStore;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.android.material.textfield.TextInputEditText;
import com.hardcodecoder.pulsemusic.R;
import com.hardcodecoder.pulsemusic.TaskRunner;
import com.hardcodecoder.pulsemusic.activities.base.ControllerActivity;
import com.hardcodecoder.pulsemusic.helper.MasterListUpdater;
import com.hardcodecoder.pulsemusic.loaders.MediaStoreHelper;
import com.hardcodecoder.pulsemusic.model.AudioFileModel;
import com.hardcodecoder.pulsemusic.model.MusicModel;
import com.hardcodecoder.pulsemusic.model.TagModel;
import com.hardcodecoder.pulsemusic.tag.TagEditorUtils;
import com.hardcodecoder.pulsemusic.tag.TagUtils;
import com.hardcodecoder.pulsemusic.tag.UpdatedMusicFetcher;
import com.hardcodecoder.pulsemusic.themes.ThemeColors;
import com.hardcodecoder.pulsemusic.utils.LogUtils;
import com.hardcodecoder.pulsemusic.utils.PulseUtil;
import com.hardcodecoder.pulsemusic.views.MediaArtImageView;

import java.io.IOException;
import java.io.OutputStream;

public class TagEditorActivity extends ControllerActivity {

    public static final String ACTION_EDIT_TAG = "EditTag";
    public static final String EXTRA_TRACK_KEY = "TrackKey";
    private static final int PICK_MEDIA_ART_REQUEST_CODE = 14;
    private static final String TAG = TagEditorActivity.class.getSimpleName();
    private static final int EDIT_REQUEST_CODE = 15;
    private final Handler mHandler = TaskRunner.getMainHandler();
    private MediaArtImageView mArtworkImageView;
    private AudioFileModel mOriginalAudio;
    private AudioFileModel mUpdatedAudio;
    private MusicModel mOriginalMusicModel;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mOriginalMusicModel = (MusicModel) getIntent().getSerializableExtra(EXTRA_TRACK_KEY);
        String action = getIntent().getAction();
        if (null == action || !action.equals(ACTION_EDIT_TAG) || null == mOriginalMusicModel) {
            finish();
        } else {
            setContentView(R.layout.activity_tag_editor);
            // Ordering matches TagModel constructor
            TextInputEditText[] inputs = new TextInputEditText[]{
                    findViewById(R.id.te_title_input),
                    findViewById(R.id.te_album_input),
                    findViewById(R.id.te_album_artist_input),
                    findViewById(R.id.te_artist_input),
                    findViewById(R.id.te_track_number_input),
                    findViewById(R.id.te_track_count_input),
                    findViewById(R.id.te_disc_number_input),
                    findViewById(R.id.te_disc_count_input),
                    findViewById(R.id.te_year_input),
                    findViewById(R.id.te_genre_input),
                    findViewById(R.id.te_lyrics_input),
            };

            mArtworkImageView = findViewById(R.id.te_track_album_art);
            mArtworkImageView.loadMediaArt(mOriginalMusicModel);
            mArtworkImageView.setOnClickListener(v -> pickArt());

            ProgressDialog loadingDialog = buildLoadingDialog(getString(R.string.metadata_loading));
            TagEditorUtils.getTagFromTrack(this, mOriginalMusicModel, audioFile -> {
                mHandler.postDelayed(loadingDialog::dismiss, 100);
                if (null == audioFile) {
                    Toast.makeText(this, getString(R.string.toast_error_loading_metadata), Toast.LENGTH_LONG).show();
                    finish();
                    return;
                }
                mOriginalAudio = audioFile;
                TagModel tagModel = mOriginalAudio.getTagModel();

                inputs[0].setText(tagModel.getTitle());
                inputs[1].setText(tagModel.getAlbum());
                inputs[2].setText(tagModel.getAlbumArtist());
                inputs[3].setText(tagModel.getArtist());
                inputs[4].setText(String.valueOf(tagModel.getTrackNumber()));
                inputs[5].setText(String.valueOf(tagModel.getTrackCount()));
                inputs[6].setText(String.valueOf(tagModel.getDiscNumber()));
                inputs[7].setText(String.valueOf(tagModel.getDiscCount()));
                inputs[8].setText(tagModel.getYear());
                inputs[9].setText(tagModel.getGenre());
                inputs[10].setText(tagModel.getLyrics());
            });

            findViewById(R.id.te_save_btn).setOnClickListener(v -> saveChanges(inputs));
            findViewById(R.id.te_close_btn).setOnClickListener(v -> finish());
            findViewById(R.id.te_cancel_btn).setOnClickListener(v -> finish());
        }
    }

    private void pickArt() {
        Intent getIntent = new Intent(Intent.ACTION_GET_CONTENT);
        getIntent.setType("image/*");

        Intent pickIntent = new Intent(Intent.ACTION_PICK);
        pickIntent.setDataAndType(MediaStore.Images.Media.EXTERNAL_CONTENT_URI, "image/*");

        Intent chooserIntent = Intent.createChooser(getIntent, getString(R.string.select_image));
        chooserIntent.putExtra(Intent.EXTRA_INITIAL_INTENTS, new Intent[]{pickIntent});

        startActivityForResult(chooserIntent, PICK_MEDIA_ART_REQUEST_CODE);
    }

    @NonNull
    private ProgressDialog buildLoadingDialog(String title) {
        // Show progress dialog
        ProgressDialog progressDialog = new ProgressDialog(this);
        progressDialog.setIndeterminate(true);
        progressDialog.setProgressStyle(ProgressDialog.STYLE_SPINNER);
        progressDialog.setTitle(R.string.progress_dialog_wait_title);
        progressDialog.setMessage(title);
        progressDialog.setProgressDrawable(new ColorDrawable(ThemeColors.getCurrentColorPrimary()));
        progressDialog.setCancelable(false);
        progressDialog.setCanceledOnTouchOutside(false);
        progressDialog.show();
        return progressDialog;
    }

    @NonNull
    private String getStringText(@NonNull TextInputEditText input) {
        if (input.getText() == null) return "";
        return input.getText().toString().trim();
    }

    private void saveChanges(@NonNull TextInputEditText[] inputs) {
        String[] finalText = new String[inputs.length];
        for (int i = 0; i < inputs.length; i++) {
            finalText[i] = getStringText(inputs[i]);
        }
        try {
            TagModel updatedTag = new TagModel(
                    finalText[0],                               // Title
                    finalText[1],                               // Album
                    finalText[2],                               // Album Artist
                    finalText[3],                               // Artist
                    TagUtils.parseIntOrDefault(finalText[4]),   // Track number
                    TagUtils.parseIntOrDefault(finalText[5]),   // Track count
                    TagUtils.parseIntOrDefault(finalText[6]),   // Disc number
                    TagUtils.parseIntOrDefault(finalText[7]),   // Disc count
                    finalText[8],                               // Year
                    finalText[9],                               // Genre
                    finalText[10]                               // Lyrics
            );

            mUpdatedAudio = new AudioFileModel(
                    mOriginalAudio.getId(),
                    mOriginalAudio.getDisplayName(),
                    mOriginalAudio.getAbsolutePath(),
                    mOriginalAudio.getMineType(),
                    updatedTag
            );

            ProgressDialog savingDialog = buildLoadingDialog(getString(R.string.metadata_saving));
            TagEditorUtils.writeTagToTrack(mUpdatedAudio, result -> {
                mHandler.postDelayed(savingDialog::dismiss, 100);
                onWriteCompleted(null != result && result);
            });

        } catch (Exception e) {
            LogUtils.logException(LogUtils.Type.IO,
                    TagEditorActivity.class.getSimpleName(),
                    "at: saveChanges()",
                    e);
        }
    }

    private void onWriteCompleted(boolean success) {
        if (success) {
            // Successfully updated tag
            // Now we need to get user permission to overwrite the original track
            if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.P) {
                if (PulseUtil.isStoragePermissionGranted(this)) {
                    performUpdateOperation();
                } else {
                    PulseUtil.getStoragePermission(this);
                }
            } else {
                // See if we can edit the original file by opening a output stream
                try {
                    OutputStream outputStream = getContentResolver().openOutputStream(mOriginalAudio.getAudioUri());
                    outputStream.close();
                    // We do have access
                    performUpdateOperation();
                } catch (SecurityException | IOException exception) {
                    if (!(exception instanceof RecoverableSecurityException)) return;

                    RecoverableSecurityException rsException = (RecoverableSecurityException) exception;

                    try {
                        IntentSender intentSender = rsException.getUserAction().getActionIntent().getIntentSender();
                        startIntentSenderForResult(intentSender, EDIT_REQUEST_CODE,
                                null,
                                0,
                                0,
                                0,
                                null);
                    } catch (IntentSender.SendIntentException intentException) {
                        LogUtils.logException(LogUtils.Type.IO, TAG, "at: onWriteCompleted()", intentException);
                    }
                }
            }
        } else {
            // Operation was not successful
            Toast.makeText(this, getString(R.string.toast_error_saving_metadata), Toast.LENGTH_LONG).show();
        }
    }

    private void performUpdateOperation() {
        ProgressDialog savingDialog = buildLoadingDialog(getString(R.string.metadata_updating));
        TagEditorUtils.updateAudioTrack(this, mUpdatedAudio, result -> {
            mHandler.postDelayed(savingDialog::dismiss, 100);
            if (null == result || !result) {
                Toast.makeText(this, getString(R.string.toast_error_updating_metadata), Toast.LENGTH_LONG).show();
                finish();
            } else if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.P) {
                MediaScannerConnection.scanFile(
                        this,
                        new String[]{MediaStoreHelper.getAbsoluteAudioFilePath(this, mUpdatedAudio.getAudioUri())},
                        new String[]{mUpdatedAudio.getMineType()},
                        (path, uri) -> onAudioFileUpdated());
            } else onAudioFileUpdated();
        });
    }

    private void onAudioFileUpdated() {
        TaskRunner.executeAsync(new UpdatedMusicFetcher(this, mUpdatedAudio), updatedModel -> {
            if (null == updatedModel) {
                // Fallback
                TagModel tag = mUpdatedAudio.getTagModel();
                updatedModel = new MusicModel(
                        mOriginalMusicModel.getId(),
                        tag.getTitle(),
                        tag.getAlbum(),
                        mOriginalMusicModel.getAlbumId(),
                        tag.getArtist(),
                        mOriginalMusicModel.getTrackPath(),
                        mOriginalMusicModel.getAlbumArtUrl(),
                        mOriginalMusicModel.getDateAdded(),
                        System.currentTimeMillis(),
                        tag.getDiscNumber(),
                        tag.getTrackNumber(),
                        mOriginalMusicModel.getTrackDuration());
            }
            MasterListUpdater.getInstance().updateTrack(mOriginalMusicModel, updatedModel);
            finish();
        });
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode != RESULT_OK) return;

        if (requestCode == EDIT_REQUEST_CODE) {
            // We have been granted the permission to delete the item
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q)
                performUpdateOperation();

        } else if (requestCode == PICK_MEDIA_ART_REQUEST_CODE && null != data) {
            Uri uri = data.getData();

            TagEditorUtils.getArtworkFromUri(this, uri, success -> {
                if (null == success || !success)
                    Toast.makeText(this, getString(R.string.toast_error_failed_loading_artwork), Toast.LENGTH_LONG).show();
                else
                    mArtworkImageView.loadAlbumArt(uri.toString(), mOriginalMusicModel.getAlbumId());
            });
        }
    }

    @Override
    protected void onDestroy() {
        TagEditorUtils.deleteTempFiles(this);
        super.onDestroy();
    }
}