/*******************************************************************************
 * This file is part of Zandy.
 *
 * Zandy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zandy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Zandy.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.gimranov.zandy.app;

import java.util.ArrayList;

import android.app.AlertDialog;
import android.app.Dialog;
import android.app.ListActivity;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.text.Editable;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.AdapterView.OnItemLongClickListener;
import android.widget.ArrayAdapter;
import android.widget.EditText;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.TextView.BufferType;
import android.widget.Toast;

import com.gimranov.zandy.app.data.Database;
import com.gimranov.zandy.app.data.Item;
import com.gimranov.zandy.app.task.APIRequest;
import com.gimranov.zandy.app.task.ZoteroAPITask;

/**
 * This Activity handles displaying and editing tags. It works almost the same as
 * ItemDataActivity, using a simple ArrayAdapter on Bundles with the tag info.
 *
 * @author ajlyon
 */
public class TagActivity extends ListActivity {

    private static final String TAG = TagActivity.class.getSimpleName();

    static final int DIALOG_TAG = 3;
    static final int DIALOG_CONFIRM_NAVIGATE = 4;

    private Item item;

    private Database db;

    protected Bundle b = new Bundle();

    /**
     * Called when the activity is first created.
     */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        db = new Database(this);
                
        /* Get the incoming data from the calling activity */
        String itemKey = getIntent().getStringExtra("com.gimranov.zandy.app.itemKey");
        Item item = Item.load(itemKey, db);
        this.item = item;

        this.setTitle(getResources().getString(R.string.tags_for_item, item.getTitle()));

        ArrayList<Bundle> rows = item.tagsToBundleArray();
        
        /* 
         * We use the standard ArrayAdapter, passing in our data as a Bundle.
         * Since it's no longer a simple TextView, we need to override getView, but
         * we can do that anonymously.
         */
        setListAdapter(new ArrayAdapter<Bundle>(this, R.layout.list_data, rows) {
            @NonNull
            @Override
            public View getView(int position, View convertView, @NonNull ViewGroup parent) {
                View row;

                // We are reusing views, but we need to initialize it if null
                if (null == convertView) {
                    LayoutInflater inflater = getLayoutInflater();
                    row = inflater.inflate(R.layout.list_data, null);
                } else {
                    row = convertView;
                }
         
        		/* Our layout has just two fields */
                TextView tvLabel = row.findViewById(R.id.data_label);
                TextView tvContent = row.findViewById(R.id.data_content);

                if (getItem(position).getInt("type") == 1)
                    tvLabel.setText(getResources().getString(R.string.tag_auto));
                else
                    tvLabel.setText(getResources().getString(R.string.tag_user));
                tvContent.setText(getItem(position).getString("tag"));

                return row;
            }
        });

        ListView lv = getListView();
        lv.setTextFilterEnabled(true);
        lv.setOnItemClickListener(new OnItemClickListener() {
            // Warning here because Eclipse can't tell whether my ArrayAdapter is
            // being used with the correct parametrization.
            @SuppressWarnings("unchecked")
            public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                // If we have a click on an entry, prompt to view that tag's items.
                ArrayAdapter<Bundle> adapter = (ArrayAdapter<Bundle>) parent.getAdapter();
                Bundle row = adapter.getItem(position);
                removeDialog(DIALOG_CONFIRM_NAVIGATE);
                TagActivity.this.b = row;
                showDialog(DIALOG_CONFIRM_NAVIGATE);
            }
        });
        
        /*
         * On long click, we bring up an edit dialog.
         */
        lv.setOnItemLongClickListener(new OnItemLongClickListener() {
            /*
             * Same annotation as in onItemClick(..), above.
             */
            @SuppressWarnings("unchecked")
            public boolean onItemLongClick(AdapterView<?> parent, View view, int position, long id) {
                // If we have a long click on an entry, show an editor
                ArrayAdapter<Bundle> adapter = (ArrayAdapter<Bundle>) parent.getAdapter();
                Bundle row = adapter.getItem(position);

                removeDialog(DIALOG_TAG);
                TagActivity.this.b = row;
                showDialog(DIALOG_TAG);
                return true;
            }
        });

    }

    @Override
    public void onDestroy() {
        if (db != null) db.close();
        super.onDestroy();
    }

    @Override
    public void onResume() {
        if (db == null) db = new Database(this);
        super.onResume();
    }

    protected Dialog onCreateDialog(int id) {
        @SuppressWarnings("unused")
        final int type = b.getInt("type");
        final String tag = b.getString("tag");
        final String itemKey = b.getString("itemKey");
        AlertDialog dialog;

        switch (id) {
        /* Simple editor for a single tag */
            case DIALOG_TAG:
                final EditText input = new EditText(this);
                input.setText(tag, BufferType.EDITABLE);

                dialog = new AlertDialog.Builder(this)
                        .setTitle(getResources().getString(R.string.tag_edit))
                        .setView(input)
                        .setPositiveButton(getResources().getString(R.string.ok), new DialogInterface.OnClickListener() {
                            @SuppressWarnings("unchecked")
                            public void onClick(DialogInterface dialog, int whichButton) {
                                Editable value = input.getText();
                                Log.d(TAG, "Got tag: " + value.toString());
                                Item.setTag(itemKey, tag, value.toString(), 0, db);
                                Item item = Item.load(itemKey, db);
                                Log.d(TAG, "Have JSON: " + item.getContent().toString());
                                ArrayAdapter<Bundle> la = (ArrayAdapter<Bundle>) getListAdapter();
                                la.clear();
                                for (Bundle b : item.tagsToBundleArray()) {
                                    la.add(b);
                                }
                                la.notifyDataSetChanged();
                            }
                        }).setNegativeButton(getResources().getString(R.string.cancel), new DialogInterface.OnClickListener() {
                            public void onClick(DialogInterface dialog, int whichButton) {
                                // do nothing
                            }
                        }).create();
                return dialog;
            case DIALOG_CONFIRM_NAVIGATE:
                dialog = new AlertDialog.Builder(this)
                        .setTitle(getResources().getString(R.string.tag_view_confirm))
                        .setPositiveButton(getResources().getString(R.string.tag_view), new DialogInterface.OnClickListener() {
                            public void onClick(DialogInterface dialog, int whichButton) {
                                Intent i = new Intent(getBaseContext(), ItemActivity.class);
                                i.putExtra("com.gimranov.zandy.app.tag", tag);
                                startActivity(i);
                            }
                        }).setNegativeButton(getResources().getString(R.string.cancel), new DialogInterface.OnClickListener() {
                            public void onClick(DialogInterface dialog, int whichButton) {
                                // do nothing
                            }
                        }).create();
                return dialog;
            default:
                Log.e(TAG, "Invalid dialog requested");
                return null;
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.zotero_menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle item selection
        switch (item.getItemId()) {
            case R.id.do_sync:
                if (!ServerCredentials.check(getApplicationContext())) {
                    Toast.makeText(getApplicationContext(), getResources().getString(R.string.sync_log_in_first),
                            Toast.LENGTH_SHORT).show();
                    return true;
                }
                Log.d(TAG, "Preparing sync requests");
                new ZoteroAPITask(getBaseContext()).execute(APIRequest.update(this.item));
                Toast.makeText(getApplicationContext(), getResources().getString(R.string.sync_started),
                        Toast.LENGTH_SHORT).show();
                return true;
            case R.id.do_new:
                Bundle row = new Bundle();
                row.putString("tag", "");
                row.putString("itemKey", this.item.getKey());
                row.putInt("type", 0);
                removeDialog(DIALOG_TAG);
                this.b = row;
                showDialog(DIALOG_TAG);
                return true;
            case R.id.do_prefs:
                startActivity(new Intent(this, SettingsActivity.class));
                return true;
            default:
                return super.onOptionsItemSelected(item);
        }
    }
}
