/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { global } from './global';
export function ngDevModeResetPerfCounters() {
    const locationString = typeof location !== 'undefined' ? location.toString() : '';
    const newCounters = {
        namedConstructors: locationString.indexOf('ngDevMode=namedConstructors') != -1,
        firstCreatePass: 0,
        tNode: 0,
        tView: 0,
        rendererCreateTextNode: 0,
        rendererSetText: 0,
        rendererCreateElement: 0,
        rendererAddEventListener: 0,
        rendererSetAttribute: 0,
        rendererRemoveAttribute: 0,
        rendererSetProperty: 0,
        rendererSetClassName: 0,
        rendererAddClass: 0,
        rendererRemoveClass: 0,
        rendererSetStyle: 0,
        rendererRemoveStyle: 0,
        rendererDestroy: 0,
        rendererDestroyNode: 0,
        rendererMoveNode: 0,
        rendererRemoveNode: 0,
        rendererAppendChild: 0,
        rendererInsertBefore: 0,
        rendererCreateComment: 0,
    };
    // Make sure to refer to ngDevMode as ['ngDevMode'] for closure.
    const allowNgDevModeTrue = locationString.indexOf('ngDevMode=false') === -1;
    global['ngDevMode'] = allowNgDevModeTrue && newCounters;
    return newCounters;
}
/**
 * This function checks to see if the `ngDevMode` has been set. If yes,
 * then we honor it, otherwise we default to dev mode with additional checks.
 *
 * The idea is that unless we are doing production build where we explicitly
 * set `ngDevMode == false` we should be helping the developer by providing
 * as much early warning and errors as possible.
 *
 * `ɵɵdefineComponent` is guaranteed to have been called before any component template functions
 * (and thus Ivy instructions), so a single initialization there is sufficient to ensure ngDevMode
 * is defined for the entire instruction set.
 *
 * When using checking `ngDevMode` on toplevel, always init it before referencing it
 * (e.g. `((typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode())`), otherwise you can
 *  get a `ReferenceError` like in https://github.com/angular/angular/issues/31595.
 *
 * Details on possible values for `ngDevMode` can be found on its docstring.
 *
 * NOTE:
 * - changes to the `ngDevMode` name must be synced with `compiler-cli/src/tooling.ts`.
 */
export function initNgDevMode() {
    // The below checks are to ensure that calling `initNgDevMode` multiple times does not
    // reset the counters.
    // If the `ngDevMode` is not an object, then it means we have not created the perf counters
    // yet.
    if (typeof ngDevMode === 'undefined' || ngDevMode) {
        if (typeof ngDevMode !== 'object') {
            ngDevModeResetPerfCounters();
        }
        return !!ngDevMode;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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