/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ɵstringify as stringify } from '@angular/core';
let _nextReferenceId = 0;
export class MetadataOverrider {
    constructor() {
        this._references = new Map();
    }
    /**
     * Creates a new instance for the given metadata class
     * based on an old instance and overrides.
     */
    overrideMetadata(metadataClass, oldMetadata, override) {
        const props = {};
        if (oldMetadata) {
            _valueProps(oldMetadata).forEach((prop) => props[prop] = oldMetadata[prop]);
        }
        if (override.set) {
            if (override.remove || override.add) {
                throw new Error(`Cannot set and add/remove ${stringify(metadataClass)} at the same time!`);
            }
            setMetadata(props, override.set);
        }
        if (override.remove) {
            removeMetadata(props, override.remove, this._references);
        }
        if (override.add) {
            addMetadata(props, override.add);
        }
        return new metadataClass(props);
    }
}
function removeMetadata(metadata, remove, references) {
    const removeObjects = new Set();
    for (const prop in remove) {
        const removeValue = remove[prop];
        if (Array.isArray(removeValue)) {
            removeValue.forEach((value) => {
                removeObjects.add(_propHashKey(prop, value, references));
            });
        }
        else {
            removeObjects.add(_propHashKey(prop, removeValue, references));
        }
    }
    for (const prop in metadata) {
        const propValue = metadata[prop];
        if (Array.isArray(propValue)) {
            metadata[prop] = propValue.filter((value) => !removeObjects.has(_propHashKey(prop, value, references)));
        }
        else {
            if (removeObjects.has(_propHashKey(prop, propValue, references))) {
                metadata[prop] = undefined;
            }
        }
    }
}
function addMetadata(metadata, add) {
    for (const prop in add) {
        const addValue = add[prop];
        const propValue = metadata[prop];
        if (propValue != null && Array.isArray(propValue)) {
            metadata[prop] = propValue.concat(addValue);
        }
        else {
            metadata[prop] = addValue;
        }
    }
}
function setMetadata(metadata, set) {
    for (const prop in set) {
        metadata[prop] = set[prop];
    }
}
function _propHashKey(propName, propValue, references) {
    const replacer = (key, value) => {
        if (typeof value === 'function') {
            value = _serializeReference(value, references);
        }
        return value;
    };
    return `${propName}:${JSON.stringify(propValue, replacer)}`;
}
function _serializeReference(ref, references) {
    let id = references.get(ref);
    if (!id) {
        id = `${stringify(ref)}${_nextReferenceId++}`;
        references.set(ref, id);
    }
    return id;
}
function _valueProps(obj) {
    const props = [];
    // regular public props
    Object.keys(obj).forEach((prop) => {
        if (!prop.startsWith('_')) {
            props.push(prop);
        }
    });
    // getters
    let proto = obj;
    while (proto = Object.getPrototypeOf(proto)) {
        Object.keys(proto).forEach((protoProp) => {
            const desc = Object.getOwnPropertyDescriptor(proto, protoProp);
            if (!protoProp.startsWith('_') && desc && 'get' in desc) {
                props.push(protoProp);
            }
        });
    }
    return props;
}
//# sourceMappingURL=data:application/json;base64,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