/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertDomNode, assertIndexInRange } from '../util/assert';
import { getInsertInFrontOfRNodeWithNoI18n, nativeInsertBefore } from './node_manipulation';
import { unwrapRNode } from './util/view_utils';
/**
 * Find a node in front of which `currentTNode` should be inserted (takes i18n into account).
 *
 * This method determines the `RNode` in front of which we should insert the `currentRNode`. This
 * takes `TNode.insertBeforeIndex` into account.
 *
 * @param parentTNode parent `TNode`
 * @param currentTNode current `TNode` (The node which we would like to insert into the DOM)
 * @param lView current `LView`
 */
export function getInsertInFrontOfRNodeWithI18n(parentTNode, currentTNode, lView) {
    const tNodeInsertBeforeIndex = currentTNode.insertBeforeIndex;
    const insertBeforeIndex = Array.isArray(tNodeInsertBeforeIndex) ? tNodeInsertBeforeIndex[0] : tNodeInsertBeforeIndex;
    if (insertBeforeIndex === null) {
        return getInsertInFrontOfRNodeWithNoI18n(parentTNode, currentTNode, lView);
    }
    else {
        ngDevMode && assertIndexInRange(lView, insertBeforeIndex);
        return unwrapRNode(lView[insertBeforeIndex]);
    }
}
/**
 * Process `TNode.insertBeforeIndex` by adding i18n text nodes.
 *
 * See `TNode.insertBeforeIndex`
 */
export function processI18nInsertBefore(renderer, childTNode, lView, childRNode, parentRElement) {
    const tNodeInsertBeforeIndex = childTNode.insertBeforeIndex;
    if (Array.isArray(tNodeInsertBeforeIndex)) {
        // An array indicates that there are i18n nodes that need to be added as children of this
        // `childRNode`. These i18n nodes were created before this `childRNode` was available and so
        // only now can be added. The first element of the array is the normal index where we should
        // insert the `childRNode`. Additional elements are the extra nodes to be added as children of
        // `childRNode`.
        ngDevMode && assertDomNode(childRNode);
        let i18nParent = childRNode;
        let anchorRNode = null;
        if (!(childTNode.type & 3 /* AnyRNode */)) {
            anchorRNode = i18nParent;
            i18nParent = parentRElement;
        }
        if (i18nParent !== null && (childTNode.flags & 2 /* isComponentHost */) === 0) {
            for (let i = 1; i < tNodeInsertBeforeIndex.length; i++) {
                // No need to `unwrapRNode` because all of the indexes point to i18n text nodes.
                // see `assertDomNode` below.
                const i18nChild = lView[tNodeInsertBeforeIndex[i]];
                nativeInsertBefore(renderer, i18nParent, i18nChild, anchorRNode, false);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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