/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
// Below are constants for LView indices to help us look up LView members
// without having to remember the specific indices.
// Uglify will inline these when minifying so there shouldn't be a cost.
export const HOST = 0;
export const TVIEW = 1;
export const FLAGS = 2;
export const PARENT = 3;
export const NEXT = 4;
export const TRANSPLANTED_VIEWS_TO_REFRESH = 5;
export const T_HOST = 6;
export const CLEANUP = 7;
export const CONTEXT = 8;
export const INJECTOR = 9;
export const RENDERER_FACTORY = 10;
export const RENDERER = 11;
export const SANITIZER = 12;
export const CHILD_HEAD = 13;
export const CHILD_TAIL = 14;
// FIXME(misko): Investigate if the three declarations aren't all same thing.
export const DECLARATION_VIEW = 15;
export const DECLARATION_COMPONENT_VIEW = 16;
export const DECLARATION_LCONTAINER = 17;
export const PREORDER_HOOK_FLAGS = 18;
export const QUERIES = 19;
/**
 * Size of LView's header. Necessary to adjust for it when setting slots.
 *
 * IMPORTANT: `HEADER_OFFSET` should only be referred to the in the `ɵɵ*` instructions to translate
 * instruction index into `LView` index. All other indexes should be in the `LView` index space and
 * there should be no need to refer to `HEADER_OFFSET` anywhere else.
 */
export const HEADER_OFFSET = 20;
/**
 * Converts `TViewType` into human readable text.
 * Make sure this matches with `TViewType`
 */
export const TViewTypeAsString = [
    'Root',
    'Component',
    'Embedded',
];
// Note: This hack is necessary so we don't erroneously get a circular dependency
// failure based on types.
export const unusedValueExportToPlacateAjd = 1;
//# sourceMappingURL=data:application/json;base64,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